package s12.mapas.valueObjects;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONString;

import s12.constantes.S12Constantes;
import s12.util.S12Utils;

import es.ejie.frmk.listeners.base.Q70ListenerUtils;

/**
 * Clase S12Recurso.
 *
 * Representa uno de los recursos usados en el visor de Google Maps del Portal
 *
 *  
 */
public class S12Recurso implements JSONString {

	private final static String URL_PORTAL = Q70ListenerUtils.getApplicationProperty("URL_PORTAL");
	private final static String MINIATURA_PORTAL = Q70ListenerUtils.getApplicationProperty("UrlFichasMiniaturasTipo");
	private final static int LONG_CODIGO_RECURSO_LINK = 10;
	private final static int SIZE_STRINGBUFFER = 300;

	/**	Cdigo asociado al recurso */
	private String codigoRecurso;

	/** Cdigo del tipo de recurso */
	private String tipoRecurso;

	/** Nombre del recurso */
	private String nombreRecurso;

	/** Cdigo del subtipo del recurso (para tipos: A1, B1, H4, I2) */
	private String subtipoRecurso;

	/** Modalidad del recurso (para tipos: A1) */
	private String modalidadRecurso;

	/** Valor de la coordenada geogrfica correspondiente a la longitud donde se encuentra el recurso */
	private String longitud;

	/** Valor de la coordenada geogrfica correspondiente a la latitud donde se encuentra el recurso */
	private String latitud;

	/** Conjunto de coordenadas con formato [[latitud,longitud],...] usado en recursos con una ruta de georeferenciacin */
	private String ruta;

	/** Conjunto de coordenadas con formato [[latitud,longitud],...] usado en recursos con un rea de georeferenciacin */
	private String area;

	/** Conjunto de coordenadas con formato [[latitud,longitud],...] usado en recursos con varios puntos de georeferenciacin */
	private String multiPunto;

	/** Tipologa del recurso */
	private String tipologia;

	/** Idioma del recurso */
	private String idioma;

	/**
	 * Constructor.
	 */
	public S12Recurso() {
	}

	/**
	 * Retorna el valor del campo codigoRecurso.
	 *
	 * @return Cdigo del recurso
	 */
	public String getCodigoRecurso() {
		return this.codigoRecurso;
	}

	/**
	 * Establece el valor del campo codigoRecurso.
	 *
	 * @param codigoRecurso Cdigo del recurso
	 */
	public void setCodigoRecurso(String codigoRecurso) {
		this.codigoRecurso = codigoRecurso;
	}

	/**
	 * Retorna el valor del campo tipoRecurso.
	 *
	 * @return Tipo de recurso
	 */
	public String getTipoRecurso() {
		return this.tipoRecurso;
	}

	/**
	 * Establece el valor del campo tipoRecurso.
	 *
	 * @param tipoRecurso Tipo de recurso
	 */
	public void setTipoRecurso(String tipoRecurso) {
		this.tipoRecurso = tipoRecurso;
	}

	/**
	 * Retorna el valor del campo nombreRecurso.
	 *
	 * @return Nombre del recurso
	 */
	public String getNombreRecurso() {
		return this.nombreRecurso;
	}

	/**
	 * Establece el valor del campo nombreRecurso.
	 *
	 * @param nombreRecurso Nombre del recurso
	 */
	public void setNombreRecurso(String nombreRecurso) {
		this.nombreRecurso = nombreRecurso;
	}

	/**
	 * Retorna el valor del campo subtipoRecurso.
	 *
	 * @return Subtipo del recurso
	 */
	public String getSubtipoRecurso() {
		return this.subtipoRecurso;
	}

	/**
	 * Establece el valor del campo subtipoRecurso.
	 *
	 * @param subtipoRecurso Subtipo del recurso
	 */
	public void setSubtipoRecurso(String subtipoRecurso) {
		this.subtipoRecurso = subtipoRecurso;
	}

	/**
	 * Retorna el valor del campo categoriaRecurso.
	 *
	 * @return Categora del recurso
	 */
	public String getModalidadRecurso() {
		return this.modalidadRecurso;
	}

	/**
	 * Establece el valor del campo categoriaRecurso.
	 *
	 * @param categoriaRecurso Categora del recurso
	 */
	public void setModalidadRecurso(String modalidadRecurso) {
		this.modalidadRecurso = modalidadRecurso;
	}

	/**
	 * Retorna el valor del campo longitud.
	 *
	 * @return Longitud del recurso
	 */
	public String getLongitud() {
		return this.longitud;
	}

	/**
	 * Establece el valor del campo longitud.
	 *
	 * @param longitud Longitud del recurso
	 */
	public void setLongitud(String longitud) {
		this.longitud = longitud;
	}

	/**
	 * Retorna el valor del campo latitud.
	 *
	 * @return Latitud del recurso
	 */
	public String getLatitud() {
		return this.latitud;
	}

	/**
	 * Establece el valor del campo latitud.
	 *
	 * @param latitud Latitud del recurso
	 */
	public void setLatitud(String latitud) {
		this.latitud = latitud;
	}

	/**
	 * @return Retorna el valor del campo ruta.
	 */
	public String getRuta() {
		return this.ruta;
	}

	/**
	 * @param ruta Establece el valor del campo ruta.
	 */
	public void setRuta(String ruta) {
		this.ruta = ruta;
	}

	/**
	 * @return Retorna el valor del campo area.
	 */
	public String getArea() {
		return this.area;
	}

	/**
	 * @param area Establece el valor del campo area.
	 */
	public void setArea(String area) {
		this.area = area;
	}

	/**
	 * @return Retorna el valor del campo multiPunto.
	 */
	public String getMultiPunto() {
		return this.multiPunto;
	}

	/**
	 * @param multiPunto Establece el valor del campo multiPunto.
	 */
	public void setMultiPunto(String multiPunto) {
		this.multiPunto = multiPunto;
	}

	/**
	 * Retorna el valor del campo tipologia.
	 *
	 * @return tipologia Tipologa del recurso
	 */
	public String getTipologia() {
		return this.tipologia;
	}

	/**
	 * Establece el valor del campo tipologia.
	 *
	 * @param tipologia Tipologa del recurso
	 */
	public void setTipologia(String tipologia) {
		this.tipologia = tipologia;
	}

	/**
	 * @return Retorna el valor del campo idioma.
	 */
	public String getIdioma() {
		return this.idioma;
	}

	/**
	 * @param idioma Establece el valor del campo idioma.
	 */
	public void setIdioma(String idioma) {
		this.idioma = idioma;
	}

	/**
	 * Implementacin de la serializacin JSON del objeto
	 *
	 * @return Serializacin JSON del objeto
	 *
	 * @see org.json.JSONString#toJSONString()
	 */
	public String toJSONString() {
		JSONObject jObject = new JSONObject();
		try {
			jObject.put("r", this.codigoRecurso);

			StringBuffer sTipoRecurso = new StringBuffer(this.tipoRecurso);
			sTipoRecurso.append("_");
			if (this.subtipoRecurso != null) {
				sTipoRecurso.append(this.subtipoRecurso);

				// Para Hoteles con salas de reuniones. Para que se muestre otro icono.
				if ("P".equals(this.subtipoRecurso) && this.modalidadRecurso != null) {
					sTipoRecurso.append("_");
					sTipoRecurso.append(this.modalidadRecurso);
				}
			}

			jObject.put("t", sTipoRecurso);
			jObject.put("m", S12Utils.encodeHtmlTag(this.nombreRecurso));
			jObject.put("y", this.latitud);
			jObject.put("x", this.longitud);
			jObject.put("p", this.ruta);
			jObject.put("s", this.area);
			jObject.put("z", this.multiPunto);
		} catch (JSONException e) {
			e.printStackTrace();
		}

		return jObject.toString();
	}
	
	/**
	 * Implementacin de la serializacin geoJSON del objeto
	 *
	 * @return Serializacin geoJSON del objeto
	 *
	 * @see org.json.JSONString#toJSONString()
	 */
	public String toGeoJSONString() {
//		JSONObject featureCollection = new JSONObject();
		//featureCollection debe ir fuera para contener los geodatos de varios recursos 
        JSONObject feature = new JSONObject();
		try {
//			featureCollection.put("type", "FeatureCollection");
//			JSONArray featureList = new JSONArray();
			
			if (this.ruta != null) {
				//Hay que diferenciar entre recurso de tipo Ruta y de tipo Etapa (ambos tienen ruta)
				if (S12Constantes.RUTAS.equals(this.tipoRecurso)){
		            feature.put("type", "Feature");
		            //vamos a representar el trazado de la ruta con un MultiLineString, y adems con un marcador de inicio.
		            //ambos elementos irn dentro de un GeometryCollection

		            
		            JSONObject geometryCollection = new JSONObject();
		            geometryCollection.put("type", "GeometryCollection");
		            JSONArray geometries = new JSONArray();
		            		            
					JSONObject point = new JSONObject();
		            point.put("type", "Point");
		            // construct a JSONArray from a string; can also use an array or list
		            JSONArray coord = new JSONArray("[" + this.longitud + "," + this.latitud + "]");
		            point.put("coordinates", coord);
		            
		            geometries.put(point);
		            
					JSONObject multiLineString = new JSONObject();
					multiLineString.put("type", "MultiLineString");
					JSONArray coordinates = new JSONArray();					
					JSONArray ruta = new JSONArray(this.ruta);
			        // El dato en BBDD se almacena en formato [Latitud, Longitud]
	                // El geoJSON necesita un formato [Longitud, Latitud] por lo que se recorre
	                // y se almacena en el formato correcto
					
					//La siguiente lnea comentada funciona en libreras de UDA, pero no vale en esta version
					//for (Object etapa : ruta.getList()) {
					//en nuestra libreria orj.json JSONArray no tiene un getList(), iteramos de otra forma:
					for(int i=0;i<ruta.length();i++){
						Object etapa = ruta.get(i);
						JSONArray coordinatesLine = new JSONArray();
						JSONArray lineaEtapa = new JSONArray(etapa.toString());
						for(int j=0;j<lineaEtapa.length();j++) { 
							Object punto = lineaEtapa.get(j);
							JSONArray puntoEtapa = new JSONArray(punto.toString());
							JSONArray coordinate = new JSONArray();
							coordinate.put(puntoEtapa.get(1));
							coordinate.put(puntoEtapa.get(0));
							coordinatesLine.put(coordinate);
						}
						//condicin para validez del GeoJSON
						if (coordinatesLine.length()>=2){
							coordinates.put(coordinatesLine);
						} 
					}
					multiLineString.put("coordinates", coordinates);
					geometries.put(multiLineString);
					//feature.put("geometry", multiLineString);
					
					geometryCollection.put("geometries", geometries);
					feature.put("geometry", geometryCollection);
					
				} else {
				//if (S12Constantes.ETAPAS.equals(this.tipoRecurso)){
				// Tratamos el resto de casos como si fuera una ETAPA, ya que se haban introducido datos de recursos G2 como si fuera una ruta.
		            feature.put("type", "Feature");
	                JSONObject lineString = new JSONObject();
	                lineString.put("type", "LineString");
	
	                JSONArray coordinates = new JSONArray();
	
	                JSONArray ruta = new JSONArray(this.ruta);
	
	                // El dato en BBDD se almacena en formato [[Latitud, Longitud],...]
	                // El geoJSON necesita un formato [[Longitud, Latitud],...] por lo que se recorre
	                // y se almacena en el formato correcto
					for(int i=0;i<ruta.length();i++){
	                	Object punto = ruta.get(i);
	                    JSONArray puntoRuta = new JSONArray(punto.toString());
	                    JSONArray coordinate = new JSONArray();
	                    coordinate.put(puntoRuta.get(1));
	                    coordinate.put(puntoRuta.get(0));
	                    
	                    coordinates.put(coordinate);
	                }
	
	                lineString.put("coordinates", coordinates);
	
	                feature.put("geometry", lineString);
                }
			} else if (this.area != null) {
				//geoJSON para area
				feature.put("type", "Feature");
	            //vamos a representar el trazado del rea con un Polygon, y adems con un marcador de inicio.
	            //ambos elementos irn dentro de un GeometryCollection

	            JSONObject geometryCollection = new JSONObject();
	            geometryCollection.put("type", "GeometryCollection");
	            JSONArray geometries = new JSONArray();
	            		            
				JSONObject point = new JSONObject();
	            point.put("type", "Point");
	            // construct a JSONArray from a string; can also use an array or list
	            JSONArray coord = new JSONArray("[" + this.longitud + "," + this.latitud + "]");
	            point.put("coordinates", coord);
	            
	            geometries.put(point);

	            
				JSONObject polygon = new JSONObject();
	            polygon.put("type", "Polygon");
	            
	            JSONArray coordinates = new JSONArray();

                JSONArray exteriorRing = new JSONArray();

                JSONArray poligono = new JSONArray(this.area);

                // El dato en BBDD se almacena en formato [[Latitud, Longitud],...]
                // El geoJSON necesita un formato [[Longitud, Latitud],...] por lo que se recorre
                // y se almacena en el formato correcto
				for(int i=0;i<poligono.length();i++){
                	Object punto = poligono.get(i);
                    JSONArray puntoPoligono = new JSONArray(punto.toString());
                    JSONArray coordinate = new JSONArray();
                    coordinate.put(puntoPoligono.get(1));
                    coordinate.put(puntoPoligono.get(0));

                    exteriorRing.put(coordinate);
                }

                // Hay que aadir el punto inicial al array para cerrar el polgono
                JSONArray puntoPoligono = new JSONArray(poligono.get(0).toString());
                JSONArray coordinate = new JSONArray();
                coordinate.put(puntoPoligono.get(1));
                coordinate.put(puntoPoligono.get(0));

                exteriorRing.put(coordinate);

                coordinates.put(exteriorRing);

                polygon.put("coordinates", coordinates);
                geometries.put(polygon);
                //feature.put("geometry", polygon);
                geometryCollection.put("geometries", geometries);
                feature.put("geometry", geometryCollection);

			} else if (this.multiPunto != null) {
				//geoJSON para multipunto
				feature.put("type", "Feature");

				JSONObject multiPoint = new JSONObject();
                multiPoint.put("type", "MultiPoint");

                JSONArray coordinates = new JSONArray();

                JSONArray multipunto = new JSONArray(this.multiPunto);

                // El dato en BBDD se almacena en formato [[Latitud, Longitud],...]
                // El geoJSON necesita un formato [[Longitud, Latitud],...] por lo que se recorre

                // y se almacena en el formato correcto
				for(int i=0;i<multipunto.length();i++){
                	Object punto = multipunto.get(i);
                    JSONArray puntoMultipunto = new JSONArray(punto.toString());
                    JSONArray coordinate = new JSONArray();
                    coordinate.put(puntoMultipunto.get(1));
                    coordinate.put(puntoMultipunto.get(0));

                    coordinates.put(coordinate);
                }

                multiPoint.put("coordinates", coordinates);

                feature.put("geometry", multiPoint);				

			} else {
				//geoJSON para un punto
	            feature.put("type", "Feature");
				
				JSONObject point = new JSONObject();
	            point.put("type", "Point");
	            // construct a JSONArray from a string; can also use an array or list
	            JSONArray coord = new JSONArray("[" + this.longitud + "," + this.latitud + "]");
	            point.put("coordinates", coord);
	            
	            feature.put("geometry", point);
	            
			}

            JSONObject properties = new JSONObject();
            properties.put("r", this.codigoRecurso);
            StringBuffer sTipoRecurso = new StringBuffer(this.tipoRecurso);
			sTipoRecurso.append("_");
			if (this.subtipoRecurso != null) {
				sTipoRecurso.append(this.subtipoRecurso);
				// Para Hoteles con salas de reuniones. Para que se muestre otro icono.
				if ("P".equals(this.subtipoRecurso) && this.modalidadRecurso != null) {
					sTipoRecurso.append("_");
					sTipoRecurso.append(this.modalidadRecurso);
				}
			}
			properties.put("t", sTipoRecurso);
			properties.put("m", S12Utils.encodeHtmlTag(this.nombreRecurso));
			properties.put("y", this.latitud);
			properties.put("x", this.longitud);
			feature.put("properties", properties);
			
		} catch (JSONException e) {
			e.printStackTrace();
		}

		return feature.toString();
	}

	/**
	 * Implementacin de la serializacin XML del objeto
	 *
	 * @return Serializacin XML del objeto
	 *
	 */
	public String toXMLString() {
		StringBuffer xml = new StringBuffer(S12Recurso.SIZE_STRINGBUFFER);

		xml.append("<Placemark>");
		xml.append("<name codrec='");
		xml.append(this.codigoRecurso);
		xml.append("' tipo='");
		xml.append(this.tipoRecurso);
		xml.append("' subtipo='");
		xml.append(this.subtipoRecurso);		
		xml.append("'><![CDATA[ ");
		xml.append(this.nombreRecurso);
		xml.append(" ]]></name>");
		xml.append("<description><![CDATA[ ");

		// Enlace al recurso del Portal
		// Ejemplo: http://turismo.euskadi.net/contenidos/a_alojamiento/0000000053_a1_rec_turismo/es_53/53-miniatura7.html
		xml.append("<iframe style=\"margin:0px;padding:0px;\" width=\"296\" heigh=\"300\" scrolling=\"no\" frameborder=\"0\" src=\"" + this.componerLinkPortal() + "\"></iframe>");

		xml.append(" ]]></description>");
		xml.append("<link>");
		xml.append("</link>");

		if (this.area != null && !"".equals(this.area)) {
			xml.append("<Polygon>");
			xml.append("<outerBoundaryIs>");
			xml.append("<LinearRing>");
			xml.append(this.obtenerCoordenadasKML(this.area));
			xml.append("</LinearRing>");
			xml.append("</outerBoundaryIs>");
			xml.append("</Polygon>");
		} else if (this.ruta != null && !"".equals(this.ruta)) {
			if (this.ruta.indexOf("[[[") != -1) {
				// Caso especial de Vas Verdes y Rutas con arrays de tres dimensiones [[[...]]]
				xml.append("<MultiGeometry>");

				String rutas[] = this.ruta.split("\\]\\],\\[\\[");

				for (int i = 0; i < rutas.length; i++) {
					xml.append("<LineString>");
					xml.append(this.obtenerCoordenadasKML(rutas[i]));
					xml.append("</LineString>");
				}

				xml.append("</MultiGeometry>");
			} else {
				// Recurso con ruta simple
				xml.append("<LineString>");
				xml.append(this.obtenerCoordenadasKML(this.ruta));
				xml.append("</LineString>");
			}
		} else if (this.multiPunto != null && !"".equals(this.multiPunto)) {
			xml.append("<MultiGeometry>");

			String coordenadas = this.multiPunto.replaceFirst("\\[\\[", "");
			coordenadas = coordenadas.replaceFirst("\\]\\]", "");
			coordenadas = coordenadas.replaceAll(",\\[", "\n");
			coordenadas = coordenadas.replaceAll("\\]", "");

			String [] arrCoordenadas = coordenadas.split("\n");
			for (int i = 0; i < arrCoordenadas.length; i++) {

				xml.append("<Point>");
				xml.append("<coordinates>");

				// Hay que invertir las coordenadas ya que el kml necesita coordenadas de
				// la forma longitud,latitud y no latitud,longitud que se manejan en la aplicacin
				String punto[] = arrCoordenadas[i].split(",");
				xml.append(punto[1]);
				xml.append(",");
				xml.append(punto[0]);

				xml.append("</coordinates>");
				xml.append("</Point>");
			}

			xml.append("</MultiGeometry>");
		} else {
			xml.append("<Point>");
			xml.append("<coordinates>");
			xml.append(this.longitud);
			xml.append(",");
			xml.append(this.latitud);
			xml.append("</coordinates>");
			xml.append("</Point>");
		}

		xml.append("</Placemark>");

		return xml.toString();
	}

	/**
	 * Convierte un array de coordenadas en formato [[latitud, longitud],...]  [[[latitud, longitud],...],...]
	 * en un formato de coordenadas de KML:
	 * <coordinates>
	 * longitud,latitud
	 * longitud,latitud
	 * ...
	 * </coordinates>
	 *
	 * @param coordenadas the coordenadas
	 *
	 * @return String
	 */
	private String obtenerCoordenadasKML(String coordenadas) {
		StringBuffer coordenadasKML = new StringBuffer(S12Recurso.SIZE_STRINGBUFFER);
		coordenadasKML.append("<coordinates>\n");

		String coordenadasTmp = coordenadas.replaceFirst("\\[\\[", "");
		coordenadasTmp = coordenadasTmp.replaceFirst("\\]\\]", "");
		coordenadasTmp = coordenadasTmp.replaceAll(",\\[", "\n");
		coordenadasTmp = coordenadasTmp.replaceAll("\\]", "");

		// Las Rutas y Vas Verdes tienen arrays de tres dimensiones [[[...]]] y hay que
		// eliminar los [ pendientes
		coordenadasTmp = coordenadasTmp.replaceAll("\\[", "");

		// Hay que invertir las coordenadas ya que el kml necesita coordenadas de
		// la forma longitud,latitud y no latitud,longitud que se manejan en la aplicacin
		String puntos[] = coordenadasTmp.split("\n");
		for (int i = 0; i < puntos.length; i++) {
			String punto[] = puntos[i].split(",");
			coordenadasKML.append(punto[1]); // longitud
			coordenadasKML.append(",");
			coordenadasKML.append(punto[0]); // latitud
			coordenadasKML.append("\n");
		}

		coordenadasKML.append("</coordinates>");
		return coordenadasKML.toString();
	}

	/**
	 * Crea la URL con el enlace a la miniatura con la descripcin de un recurso
	 *
	 * http://www.turismoa.euskadi.net/contenidos/a_alojamiento/0000000053_a1_rec_turismo/es_53/53-miniatura7.html
	 *
	 * @return String
	 */
	private String componerLinkPortal() {
		StringBuffer sCodRecursoLink = new StringBuffer(this.codigoRecurso);

		while (sCodRecursoLink.length() < S12Recurso.LONG_CODIGO_RECURSO_LINK) {
			sCodRecursoLink.insert(0, "0");
		}

		StringBuffer link = new StringBuffer(S12Recurso.URL_PORTAL);
		link.append("/contenidos/");
		link.append(this.tipologia);
		link.append("/");
		link.append(sCodRecursoLink.toString());
		link.append("_");
		link.append(this.tipoRecurso.toLowerCase());
		link.append("_rec_turismo/");
		link.append(this.idioma);
		link.append("_");
		link.append(this.codigoRecurso);
		link.append("/");
		link.append(this.codigoRecurso);
		link.append(S12Recurso.MINIATURA_PORTAL);

		return link.toString();
	}

	/**
	 * Equals.
	 *
	 * @param o the o
	 *
	 * @return true, if equals
	 *
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof S12Recurso)) {
			return false;
		}
		S12Recurso other = (S12Recurso) o;
		return other.getCodigoRecurso().equals(this.codigoRecurso);
	}


}
