package s12.mapas.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Locale;

import javax.naming.NamingException;

import org.json.JSONException;
import org.json.JSONObject;

import s12.constantes.S12Constantes;
import s12.mapas.valueObjects.S12Municipio;
import s12.mapas.valueObjects.S12Recurso;
import s12.mapas.valueObjects.S12RecursoGastronomika;
import s12.util.S12ConnectionManager;
import s12.util.S12Utils;

import com.ejie.r01m.objects.searchengine.query.R01MQueryObject;
import com.ejie.r01m.objects.searchengine.query.R01MSearchedMetaData;

import es.ejie.frmk.Q70Factoria;
import es.ejie.frmk.infraestructura.traza.Q70TraceLevel;
import es.ejie.frmk.listeners.base.Q70ListenerUtils;

/**
 * The Class S12MapasDAO.
 *
 *  
 */
public class S12MapasDAO {

    /** The Constant COMPONENTE. */
    private static final String COMPONENTE = "MAPAS";

    /** Operadores de lote 3 */
    private static final String EQ = "EQ";
    private static final String LIKE = "LIKE";
    private static final String IN = "IN";
    private static final String GT = "GT";
    private static final String GTE = "GTE";
    private static final String LT = "LT";
    private static final String LTE = "LTE";

    /**
     * Obtiene aquellos recursos que cumplan los criterios de bsqueda especificados en el filtro
     *
     * @param lat1 String Latitud de la esquina inferior izquierda de la vista
     * @param lng1 String Longitud de la esquina inferior izquierda de la vista
     * @param lat2 String Latitud de la esquina superior derecha de la vista
     * @param lng2 String Longitud de la esquina superior derecha de la vista
     * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
     * 	Formato: tipoRecurso_subTipoRecurso_Categoria. Ejemplo: A1_H_1,B1_1,C2_
     * @param idioma String Idioma de la vista
     *
     * @return Array de objetos S12Recurso
     *
     * @throws S12MapasDaoException Excepcin
     */
    public S12Recurso[] filtrarRecursos(String lat1, String lng1, String lat2,String lng2,
        String tiposRecursos, String idioma) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
	    ResultSet rs = null;
	    ArrayList<S12Recurso> recursosList = new ArrayList<S12Recurso>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();
    
	    try {
	    	
	        String[] tiposRecursoArray = tiposRecursos.split(",");
	    	
	        conn = S12ConnectionManager.getConnection();
	
	        StringBuffer sql = new StringBuffer();
	        sql.append(" SELECT CODIGORECURSO, COTIPREC, IDIOMA, NOMBRERECURSO, CODSUBTIPO, MODALIDAD, LONGITUD, LATITUD, AREA, RUTA, MULTIPUNTO, TYPO, IMPORTANCIA ");
	        sql.append(" FROM CONSULTAMAT_RECURSOS WHERE (");
	       
	        
	        // Tipos, subtipos y categora de los recursos
	        for (int i = 0; i < tiposRecursoArray.length; i++) {
	        	String[] tipoRecurso = tiposRecursoArray[i].split("_");
	
	            String codigoTipoRecurso = tipoRecurso[0];
	                        
	            if (i > 0) {
	                sql.append(" OR ");
	            }
	            
	            
	            if (tiposRecursoArray[i].equals("camino_santiago_ignaciano") || tiposRecursoArray[i].equals("camino_santiago_interior") || tiposRecursoArray[i].equals("camino_santiago_costa") || tiposRecursoArray[i].equals("camino_santiago_entero")){
	        		sql.append("( CODIGORECURSO IN (");       		
	        		//TODO: revisar esto, no puede star bien, con esas comillas
	        		if (tiposRecursoArray[i].equals("camino_santiago_ignaciano")) {
	        			sql.append("'");
	        			sql.append(Q70ListenerUtils.getApplicationProperty( "ruta.ignaciana"));
	        			sql.append("'");
	        		}
	        		if (tiposRecursoArray[i].equals("camino_santiago_interior")) {
	        			sql.append("'");
	        			sql.append(Q70ListenerUtils.getApplicationProperty( "ruta.santiago.interior"));
	        			sql.append("'");
	        		}
	        		if (tiposRecursoArray[i].equals("camino_santiago_costa")) {
	        			sql.append("'");
	        			sql.append(Q70ListenerUtils.getApplicationProperty( "ruta.santiago.costa"));
	        			sql.append("'");
	        		}
	        		sql.append("))"); 
	        	}else{
	            
	            
	            sql.append("(");
	
	            // Cdigo de Tipo de Recurso (COTIPREC)
	            sql.append("COTIPREC = ?");
	            params.add(codigoTipoRecurso);
	            tiposSQL.add(Types.VARCHAR);
	
	            // Subtipo de Recurso (TIPORECURSO)
	            if (tipoRecurso.length > 1) {
	                String subTipoRecurso = tipoRecurso[1];
	
	                sql.append(" AND CODSUBTIPO = ?");
	                params.add(subTipoRecurso);
	                tiposSQL.add(Types.VARCHAR);
	            }
	
	            // modalidad (MODALIDAD)
	            if (tipoRecurso.length > 2) {
	                String modalidad = tipoRecurso[2];
	                sql.append(" AND MODALIDAD = ?");
	                params.add(modalidad);
	                tiposSQL.add(Types.VARCHAR);
	               // if(S12Constantes.EXPERIENCIAS.equals(codigoTipoRecurso)){
	                 //   sql.append(" AND (IMPORTANCIA IS NULL OR IMPORTANCIA > 1)");
	                //}
	            }
	            sql.append(")");
	        	}
	            
	        }
	
	        // Idioma
	        sql.append(") AND IDIOMA = ? ");
	        params.add(idioma);
	        tiposSQL.add(Types.VARCHAR);
	
	        Double valor = null;
	        // Esquina abajo izquierda
	        sql.append(" AND LONGITUD > ?");
	        valor = new Double(lng1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD > ?");
	        valor = new Double(lat1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	 
	        // Esquina arriba derecha
	        sql.append(" AND LONGITUD < ?");
	        valor = new Double(lng2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD < ?");
	        valor = new Double(lat2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	
	        // Ordenacin
	        sql.append(" ORDER BY COTIPREC, CODSUBTIPO, MODALIDAD, NOMBRERECURSO ASC ");

            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarRecursos: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarRecursos: Params: " + params.toString());
            }
	        
			prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();
	        
	        while (rs.next()) {
                S12Recurso recurso = new S12Recurso();

                recurso.setCodigoRecurso(rs.getString("CODIGORECURSO"));
                recurso.setTipoRecurso(rs.getString("COTIPREC"));
                recurso.setIdioma(rs.getString("IDIOMA"));
                recurso.setNombreRecurso(rs.getString("NOMBRERECURSO"));
                String tipoRecurso = rs.getString("CODSUBTIPO");               
                recurso.setSubtipoRecurso(tipoRecurso);
                recurso.setModalidadRecurso(rs.getString("MODALIDAD"));
                recurso.setLongitud(rs.getString("LONGITUD"));
                recurso.setLatitud(rs.getString("LATITUD"));
                recurso.setRuta(rs.getString("RUTA"));
                recurso.setArea(rs.getString("AREA"));
                recurso.setMultiPunto(rs.getString("MULTIPUNTO"));
                recurso.setTipologia(rs.getString("TYPO"));

                if (!recursosList.contains(recurso)) {
                    recursosList.add(recurso);
                }
	        }
	    } catch (NamingException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } catch (SQLException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } finally {
	        S12ConnectionManager.close(rs);
	        S12ConnectionManager.close(prepStmt);
	        S12ConnectionManager.close(conn);
	    }
	
	    S12Recurso[] recursoArray = new S12Recurso[recursosList.size()];
	    recursosList.toArray( recursoArray );
	    return recursoArray;
    }

   
    /**
     * Obtiene aquellos recursos de Euskadi gastronomika que cumplan los criterios de
     * bsqueda especificados en el filtro
     *
     * @param lat1 String Latitud de la esquina inferior izquierda de la vista
     * @param lng1 String Longitud de la esquina inferior izquierda de la vista
     * @param lat2 String Latitud de la esquina superior derecha de la vista
     * @param lng2 String Longitud de la esquina superior derecha de la vista
     * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
     * 	Formato: tipoRecurso. Ejemplo: Z1,Z2
     * @param idioma String Idioma de la vista
     *
     * @return Array de objetos S12RecursoGastronomika
     *
     * @throws S12MapasDaoException Excepcin
     */
    public S12RecursoGastronomika[] filtrarRecursosGastronomika(String lat1, String lng1, String lat2,String lng2,
        String tiposRecursos, String idioma) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;
        ArrayList<S12RecursoGastronomika> recursosList = new ArrayList<S12RecursoGastronomika>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();

        try {
            String[] tiposRecursoArray = tiposRecursos.split(",");

            conn = S12ConnectionManager.getConnection();

            StringBuffer sql = new StringBuffer();
            sql.append(" SELECT CODIGORECURSO, COTIPREC, IDIOMA, NOMBRERECURSO, TIPORECURSO, ");
            sql.append(" LONGITUD, LATITUD, TYPO FROM GK_CONSULTAMAT_RECURSOS WHERE TIPORECURSO IN(");

            // Tipos de los recursos
            for (int i = 0; i < tiposRecursoArray.length; i++) {
            	sql.append("?");
            	if (i < (tiposRecursoArray.length - 1)) {
    				sql.append(", ");
    			}
    	        params.add(tiposRecursoArray[i]);
    	        tiposSQL.add(Types.VARCHAR);
            }

            // Idioma
            sql.append(") AND IDIOMA = ? ");
	        params.add(idioma);
	        tiposSQL.add(Types.VARCHAR);

	        Double valor = null;
	        // Esquina abajo izquierda
	        sql.append(" AND LONGITUD > ?");
	        valor = new Double(lng1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD > ?");
	        valor = new Double(lat1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	 
	        // Esquina arriba derecha
	        sql.append(" AND LONGITUD < ?");
	        valor = new Double(lng2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD < ?");
	        valor = new Double(lat2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);

            // Ordenacin
            sql.append(" ORDER BY COTIPREC, TIPORECURSO, NOMBRERECURSO ASC ");

            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarRecursosGastronomika: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarRecursosGastronomika: Params: " + params.toString());
            }
            
			prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();

            while (rs.next()) {
                S12RecursoGastronomika recurso = new S12RecursoGastronomika();

                recurso.setCodigoRecurso(rs.getString("CODIGORECURSO"));
                recurso.setTipoRecurso(rs.getString("COTIPREC"));
                recurso.setIdioma(rs.getString("IDIOMA"));
                recurso.setNombreRecurso(rs.getString("NOMBRERECURSO"));
                recurso.setTipoRecursoGastronomika(rs.getString("TIPORECURSO"));
                recurso.setLongitud(rs.getString("LONGITUD"));
                recurso.setLatitud(rs.getString("LATITUD"));
                recurso.setTipologia(rs.getString("TYPO"));

                if (!recursosList.contains(recurso)) {
                    recursosList.add(recurso);
                }
            }
        } catch (NamingException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                    Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
                }
          throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
        } catch (SQLException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                    Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
                }
          throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
        } finally {
            S12ConnectionManager.close(rs);
            S12ConnectionManager.close(prepStmt);
            S12ConnectionManager.close(conn);
        }

        S12RecursoGastronomika[] recursoArray = new S12RecursoGastronomika[recursosList.size()];
        recursosList.toArray( recursoArray );
        return recursoArray;
    }

    /**
     * Obtiene aquellos municipios que cumplan los criterios de bsqueda especificados en el filtro
     *
     * @param lat1 String Latitud de la esquina inferior izquierda de la vista
     * @param lng1 String Longitud de la esquina inferior izquierda de la vista
     * @param lat2 String Latitud de la esquina superior derecha de la vista
     * @param lng2 String Longitud de la esquina superior derecha de la vista
     * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
     * 	Formato: tipoRecurso_subTipoRecurso_Categoria. Ejemplo: A1_H_1,B1_1,C2_
     * @param idioma String Idioma de la vista
     *
     * @return Array de objetos S12Municipio
     *
     * @throws S12MapasDaoException Excepcin
     */
    public S12Municipio[] filtrarMunicipios(String lat1, String lng1, String lat2,String lng2,
        String tiposRecursos, String idioma) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;
        ArrayList<S12Municipio> municipiosList = new ArrayList<S12Municipio>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();
        
	    String[] tiposRecursoArray = tiposRecursos.split(",");
	    //Quito los que van a ir por Rutas, ya que se buscan por cdigo y no por subtipo y los or de la query se descontrolan
//	    for (int i = 0; i < tiposRecursoArray.length; i++) {
//	    	String recurso =  tiposRecursoArray[i];
//	    	if (recurso.equals("camino_santiago_ignaciano") || recurso.equals("camino_santiago_interior") || recurso.equals("camino_santiago_costa") || recurso.equals("camino_santiago_entero")){
//			 	//	
//	    	}
//	    }
    
	    try {
	        conn = S12ConnectionManager.getConnection();
	
	        StringBuffer sql = new StringBuffer();
	        sql.append(" SELECT DISTINCT(COREC), NOMUN, LATITUD, LONGITUD ");
	        sql.append(" FROM CONSULTAMAT_REC_MUNICIPIOS WHERE (");
	
	        // Tipos, subtipos y categora de los recursos
	        
	        for (int i = 0; i < tiposRecursoArray.length; i++) {
	        	
	        	String recurso =  tiposRecursoArray[i];
	        	//Si viene algo relacionado con los caminos de Santiago no puedo buscar ni por tipos ni por subtipos
	        	if (i > 0 ) {
	        		sql.append(" OR ");
		        }
	        	if (recurso.equals("camino_santiago_ignaciano") || recurso.equals("camino_santiago_interior") || recurso.equals("camino_santiago_costa") || recurso.equals("camino_santiago_entero")){
	        		sql.append("( COREC IN (");
	        		//los COREC vendrn en un string que representa una lista de valores separados por comas
	        		if (recurso.equals("camino_santiago_ignaciano")) {
	        			String listaValores = Q70ListenerUtils.getApplicationProperty( "municipios.ignaciana");
	        			sql.append(this.parametrizarValoresIN(listaValores,params,tiposSQL));
	        		}
	        		if (recurso.equals("camino_santiago_interior")) {
	        			String listaValores = Q70ListenerUtils.getApplicationProperty( "municipios.santiago.interior");
	        			sql.append(this.parametrizarValoresIN(listaValores,params,tiposSQL));
	        		}
	        		if (recurso.equals("camino_santiago_costa")) {
	        			String listaValores = Q70ListenerUtils.getApplicationProperty( "municipios.santiago.costa");
	        			sql.append(this.parametrizarValoresIN(listaValores,params,tiposSQL));
	        		}
	        		sql.append("))"); 
	        	}else{
	        		String[] tipoRecurso = tiposRecursoArray[i].split("_");
	 	            String codigoTipoRecurso = tipoRecurso[0];
	 	
	 	            sql.append("(");
	 	
	 	            // Cdigo de Tipo de Recurso (COTIPREC)
	 	            sql.append("COTIPREC = ?");
        	        params.add(codigoTipoRecurso);
        	        tiposSQL.add(Types.VARCHAR);
	 	
	 	            // Subtipo de Recurso (TIPORECURSO) 89.0.1.176
	 	            if (tipoRecurso.length > 1) {
	 	                String subTipoRecurso = tipoRecurso[1];
	 	                sql.append(" AND CODSUBTIPO = ?");
	        	        params.add(subTipoRecurso);
	        	        tiposSQL.add(Types.VARCHAR);
	 	            }
	 	
	 	            // Categora (CATEGORIARECURSO)
	 	            if (tipoRecurso.length > 2 ) {
	 	                String modalidad = tipoRecurso[2];
	 	                sql.append(" AND MODALIDAD = ?");
	        	        params.add(modalidad);
	        	        tiposSQL.add(Types.VARCHAR);
	 	              }
	 	           
	 	           sql.append(")");  
	 	         
	        	}
	        }    
	
	        // Idioma
	        sql.append(") AND IDIOMA = ? ");
	        params.add(idioma);
	        tiposSQL.add(Types.VARCHAR);
	
	        Double valor = null;
	        // Esquina abajo izquierda
	        sql.append(" AND LONGITUD > ?");
	        valor = new Double(lng1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD > ?");
	        valor = new Double(lat1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	 
	        // Esquina arriba derecha
	        sql.append(" AND LONGITUD < ?");
	        valor = new Double(lng2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD < ?");
	        valor = new Double(lat2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarMunicipios: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarMunicipios: Params: " + params.toString());
            }

            prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();
	
	        while (rs.next()) {
	                S12Municipio municipio = new S12Municipio();
	
	                municipio.setCodigoRecurso(rs.getString("COREC"));
	                municipio.setNombreMunicipio(rs.getString("NOMUN"));
	                municipio.setLongitud(rs.getString("LONGITUD"));
	                municipio.setLatitud(rs.getString("LATITUD"));
	
	                municipiosList.add(municipio);
	        }
	    } catch (NamingException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarMunicipios: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } catch (SQLException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarMunicipios: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } finally {
	        S12ConnectionManager.close(rs);
	        S12ConnectionManager.close(prepStmt);
	        S12ConnectionManager.close(conn);
	    }
	
	    S12Municipio[] municipioArray = new S12Municipio[municipiosList.size()];
	    municipiosList.toArray(municipioArray);
	    return municipioArray;
    }

    /**
     * Obtiene aquellos municipios de Euskadi Gastronomika que cumplan los criterios de
     * bsqueda especificados en el filtro
     *
     * @param lat1 String Latitud de la esquina inferior izquierda de la vista
     * @param lng1 String Longitud de la esquina inferior izquierda de la vista
     * @param lat2 String Latitud de la esquina superior derecha de la vista
     * @param lng2 String Longitud de la esquina superior derecha de la vista
     * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
     * 	Formato: tipoRecurso. Ejemplo: Z1,Z2
     * @param idioma String Idioma de la vista
     *
     * @return Array de objetos S12Municipio
     *
     * @throws S12MapasDaoException Excepcin
     */
    public S12Municipio[] filtrarMunicipiosGastronomika(String lat1, String lng1, String lat2,String lng2,
        String tiposRecursos, String idioma) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;
        ArrayList<S12Municipio> municipiosList = new ArrayList<S12Municipio>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();

        String[] tiposRecursoArray = tiposRecursos.split(",");

        try {
            conn = S12ConnectionManager.getConnection();

            StringBuffer sql = new StringBuffer();
            sql.append(" SELECT DISTINCT(COREC), NOMUN, LATITUD, LONGITUD ");
            sql.append(" FROM GK_CONSULTAMAT_REC_MUNICIPIOS WHERE TIPORECURSO IN (");

            // Tipos de los recursos
            for (int i = 0; i < tiposRecursoArray.length; i++) {
                if (i > 0) {
                    sql.append(",");
                }
                sql.append("?");
    	        params.add(tiposRecursoArray[i]);
    	        tiposSQL.add(Types.VARCHAR);
            }

            // Idioma
            sql.append(") AND IDIOMA = ? ");
	        params.add(idioma);
	        tiposSQL.add(Types.VARCHAR);

	        Double valor = null;
	        // Esquina abajo izquierda
	        sql.append(" AND LONGITUD > ?");
	        valor = new Double(lng1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD > ?");
	        valor = new Double(lat1);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	 
	        // Esquina arriba derecha
	        sql.append(" AND LONGITUD < ?");
	        valor = new Double(lng2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);
	        sql.append(" AND LATITUD < ?");
	        valor = new Double(lat2);
	        params.add(valor);
	        tiposSQL.add(Types.DOUBLE);

            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarMunicipiosGastronomika: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarMunicipiosGastronomika: Params: " + params.toString());
            }
	        
			prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();

            while (rs.next()) {
                S12Municipio municipio = new S12Municipio();

                municipio.setCodigoRecurso(rs.getString("COREC"));
                municipio.setNombreMunicipio(rs.getString("NOMUN"));
                municipio.setLongitud(rs.getString("LONGITUD"));
                municipio.setLatitud(rs.getString("LATITUD"));

                municipiosList.add(municipio);
            }
        } catch (NamingException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                    Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarMunicipiosGastronomika: Exception: " + e.getMessage());
                }
          throw new S12MapasDaoException("S12MapasDAO::filtrarMunicipiosGastronomika: Exception ", e );
        } catch (SQLException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                    Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarMunicipiosGastronomika: Exception: " + e.getMessage());
                }
          throw new S12MapasDaoException("S12MapasDAO::filtrarMunicipiosGastronomika: Exception ", e );
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
        } finally {
            S12ConnectionManager.close(rs);
            S12ConnectionManager.close(prepStmt);
            S12ConnectionManager.close(conn);
        }

        S12Municipio[] municipioArray = new S12Municipio[municipiosList.size()];
        municipiosList.toArray(municipioArray);
        return municipioArray;
    }

    /**
     * Obtiene los datos del recurso que cumple los criterios de bsqueda especificados en el filtro.
     *
     * @param codigoRecurso Cdigo del recurso
     * @param idioma String Idioma de la vista
     *
     * @return Recurso
     *
     * @throws S12MapasDaoException Excepcin
     */
    public S12Recurso obtenerRecurso(String codigoRecurso, String idioma) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;

        S12Recurso recurso = null;

	    try {
	        conn = S12ConnectionManager.getConnection();
	
	        StringBuffer sql = new StringBuffer();
	        sql.append(" SELECT CODIGORECURSO, NOMBRERECURSO, CODSUBTIPO, COTIPREC, IDIOMA, MODALIDAD, LONGITUD, LATITUD, AREA, RUTA, MULTIPUNTO, TYPO ");
	        sql.append(" FROM CONSULTAMAT_RECURSOS WHERE ");

	        // Cdigo de recurso
	        sql.append(" CODIGORECURSO = ? ");

	        // Idioma
	        sql.append(" AND IDIOMA = ? ");
	
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarMunicipiosGastronomika: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarMunicipiosGastronomika: Params: " + codigoRecurso + " , " + idioma);
            }

            prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			prepStmt.setString(index++, codigoRecurso);	
			prepStmt.setString(index++, idioma);	
			
			rs = prepStmt.executeQuery();
	
	        if (rs.next()) {
	            recurso = new S12Recurso();
	                recurso.setCodigoRecurso(rs.getString("CODIGORECURSO"));
	                recurso.setTipoRecurso(rs.getString("COTIPREC"));
	                recurso.setIdioma(rs.getString("IDIOMA"));
	                recurso.setNombreRecurso(rs.getString("NOMBRERECURSO"));
	                recurso.setSubtipoRecurso(rs.getString("CODSUBTIPO"));
	                recurso.setModalidadRecurso(rs.getString("MODALIDAD"));
	                recurso.setLongitud(rs.getString("LONGITUD"));
	                recurso.setLatitud(rs.getString("LATITUD"));
	                recurso.setRuta(rs.getString("RUTA"));
	                recurso.setArea(rs.getString("AREA"));
	                recurso.setMultiPunto(rs.getString("MULTIPUNTO"));
	                recurso.setTipologia(rs.getString("TYPO"));
	        }
	    } catch (NamingException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::obtenerRecurso: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::obtenerRecurso: Exception ", e );
	    } catch (SQLException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::obtenerRecurso: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::obtenerRecurso: Exception ", e );
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } finally {
	        S12ConnectionManager.close(rs);
	        S12ConnectionManager.close(prepStmt);
	        S12ConnectionManager.close(conn);
	    }
	
	    return recurso;
    }

    /**
     * Obtiene los datos de los recursos que cumplen los criterios de bsqueda especificados en el filtro.
     *
     * @param codigosRecursos Cdigos de los recurso separados por ","
     * @param idioma String Idioma de la vista
     *
     * @return Array de recursos
     *
     * @throws S12MapasDaoException Excepcin
     */
    public S12Recurso[] obtenerRecursos(String codigosRecursos, String idioma) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;
        ArrayList<S12Recurso> recursosList = new ArrayList<S12Recurso>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();

	    try {
	        conn = S12ConnectionManager.getConnection();
	
	        StringBuffer sql = new StringBuffer();
	        sql.append(" SELECT CODIGORECURSO, NOMBRERECURSO, CODSUBTIPO, COTIPREC, IDIOMA, MODALIDAD, LONGITUD, LATITUD, AREA, RUTA, MULTIPUNTO, TYPO ");
	        sql.append(" FROM CONSULTAMAT_RECURSOS WHERE ");
	
	        // Cdigos de recursos
	        String[] arrayCodigosRecursos = codigosRecursos.split(",");
	
	        sql.append(" CODIGORECURSO IN (");
	
	        for (int i = 0; i < arrayCodigosRecursos.length; i++) {
	            /**
	             * ORACLE limita a un mximo de 1000 elementos para una lista como las que usa el operador IN.
	             * Para evitar el error cuando se supera ese lmite, separamos la lista en varias condiciones con OR, ya que podra superarse este nmero en algn caso (recursos tipo A1 sin filtro son mil y pico)
	             */
	        	if (i>0 && ((i % 999)==0)) {
	        		sql.append(") OR CODIGORECURSO IN (");
	        	} else if (i>0) {
	        		sql.append(", ");
	        	}
    			sql.append("?");
        		Long corec = new Long(arrayCodigosRecursos[i]);
    	        params.add(corec);
    	        tiposSQL.add(Types.NUMERIC);
	        }
	        sql.append(") ");
	
	        // Idioma
	        sql.append(" AND IDIOMA = ? ");
	        params.add(idioma);
	        tiposSQL.add(Types.VARCHAR);

            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::obtenerRecursos: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::obtenerRecursos: Params: " + params.toString());
            }
	        
			prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();
	
	        while (rs.next()) {
	            S12Recurso recurso = new S12Recurso();
	
	            recurso = new S12Recurso();
	                recurso.setCodigoRecurso(rs.getString("CODIGORECURSO"));
	                recurso.setTipoRecurso(rs.getString("COTIPREC"));
	                recurso.setIdioma(rs.getString("IDIOMA"));
	                recurso.setNombreRecurso(rs.getString("NOMBRERECURSO"));
	                recurso.setSubtipoRecurso(rs.getString("CODSUBTIPO"));
	                recurso.setModalidadRecurso(rs.getString("MODALIDAD"));
	                recurso.setLongitud(rs.getString("LONGITUD"));
	                recurso.setLatitud(rs.getString("LATITUD"));
	                recurso.setRuta(rs.getString("RUTA"));
	                recurso.setArea(rs.getString("AREA"));
	                recurso.setMultiPunto(rs.getString("MULTIPUNTO"));
	                recurso.setTipologia(rs.getString("TYPO"));
	
	                if (!recursosList.contains(recurso)) {
	                    recursosList.add(recurso);
	                }
	        }
	    } catch (NamingException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::obtenerRecurso: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::obtenerRecurso: Exception ", e );
	    } catch (SQLException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::obtenerRecurso: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::obtenerRecurso: Exception ", e );
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } finally {
	        S12ConnectionManager.close(rs);
	        S12ConnectionManager.close(prepStmt);
	        S12ConnectionManager.close(conn);
	    }
	
	    S12Recurso[] recursoArray = new S12Recurso[recursosList.size()];
	    recursosList.toArray( recursoArray );
	    return recursoArray;
    }

    /**
     * Obtiene todos los recursos importantes
     *
     * @param idioma String Idioma de la vista
     *
     * @param importancia String importancia
     *
     * @return Array de objetos S12Recurso
     *
     * @throws S12MapasDaoException Excepcin
     */
    public S12Recurso[] filtrarRecursosImportantes(String idioma, String importancia) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;
        ArrayList<S12Recurso> recursosList = new ArrayList<S12Recurso>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();

	    try {
	    	conn = S12ConnectionManager.getConnection();

	        StringBuffer sql = new StringBuffer();
	        sql.append(" SELECT CODIGORECURSO, COTIPREC, IDIOMA, NOMBRERECURSO, CODSUBTIPO, MODALIDAD, LONGITUD, LATITUD, AREA, RUTA, MULTIPUNTO, TYPO ");
	        sql.append(" FROM CONSULTAMAT_RECURSOS WHERE ");

	        // Cdigos de importancia
	        String[] arrayCodigosImportancia = importancia.split(",");

	        sql.append("IMPORTANCIA IN (");

	        for (int i = 0; i < arrayCodigosImportancia.length; i++) {
    			if (i>0){
    				sql.append(", ");    				
    			}
	        	sql.append("?");
        		Long codImportancia = new Long(arrayCodigosImportancia[i]);
    	        params.add(codImportancia);
    	        tiposSQL.add(Types.NUMERIC);
	        }
	        sql.append(") ");

	        sql.append("AND IDIOMA = ? ");
	        params.add(idioma);
	        tiposSQL.add(Types.VARCHAR);
	        // Ordenacin
	        sql.append(" ORDER BY COTIPREC, CODSUBTIPO, MODALIDAD, NOMBRERECURSO ASC ");
	
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarRecursosImportantes: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarRecursosImportantes: Params: " + params.toString());
            }

            prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();

			while (rs.next()) {
                S12Recurso recurso = new S12Recurso();

                recurso.setCodigoRecurso(rs.getString("CODIGORECURSO"));
                recurso.setTipoRecurso(rs.getString("COTIPREC"));
                recurso.setIdioma(rs.getString("IDIOMA"));
                recurso.setNombreRecurso(rs.getString("NOMBRERECURSO"));
                recurso.setSubtipoRecurso(rs.getString("CODSUBTIPO"));
                recurso.setModalidadRecurso(rs.getString("MODALIDAD"));
                recurso.setLongitud(rs.getString("LONGITUD"));
                recurso.setLatitud(rs.getString("LATITUD"));
                recurso.setRuta(rs.getString("RUTA"));
                recurso.setArea(rs.getString("AREA"));
                recurso.setMultiPunto(rs.getString("MULTIPUNTO"));
                recurso.setTipologia(rs.getString("TYPO"));

                if (!recursosList.contains(recurso)) {
                    recursosList.add(recurso);
                }
			}
	    } catch (NamingException e) {
	    	if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursosImportantes: Exception: " + e.getMessage());
            }
	    	throw new S12MapasDaoException("S12MapasDAO::filtrarRecursosImportantes: Exception ", e );
	    } catch (SQLException e) {
	    	if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursosImportantes: Exception: " + e.getMessage());
            }
	    	throw new S12MapasDaoException("S12MapasDAO::filtrarRecursosImportantes: Exception ", e );
	    } catch (NullPointerException e) {
	    	if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
            }
	    	throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
	    } finally {
	        S12ConnectionManager.close(rs);
	        S12ConnectionManager.close(prepStmt);
	        S12ConnectionManager.close(conn);
	    }

	    S12Recurso[] recursoArray = new S12Recurso[recursosList.size()];
	    recursosList.toArray( recursoArray );
	    return recursoArray;
    }

    /**
     * Obtiene todos los cdigos de recurso que que cumplen con las condiciones de la
     * consulta pasada por parmetro.
     *
     * @param language Idioma
     * @param queryObject Objeto R01MQueryObject con toda la informacin de la consulta
     * 
     * @return Array de String
     * 
     * @throws S12MapasDaoException Excepcin
     */
    public String[] filtrarQueryRecursos(String language, R01MQueryObject queryObject) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;
        ArrayList<String> codigosRecursoList = new ArrayList<String>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();

        try {
            conn = S12ConnectionManager.getConnection();

            if (queryObject == null || queryObject.getMetaData() == null) {
            	if (Q70Factoria.getTraza().isTraceActive()) {
                    Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarQueryRecursos: queryObject: " + queryObject);
                }
            	
            	return new String[0];
            }

            StringBuffer sql = new StringBuffer();
            sql.append(" SELECT COREC FROM CONSULTAMAT_RECURSOS_PM WHERE LANGUAGE = ? ");
            params.add(language);
            tiposSQL.add(Types.VARCHAR);

            for (int i = 0; i < queryObject.getMetaData().size(); i++) {
                R01MSearchedMetaData searchedMetadata = (R01MSearchedMetaData) queryObject.getMetaData().get(i);

                // Comprobamos si el metadato debe ignorarse, si no lo aadimos a la consulta
                if (!"".equals(S12Constantes.METADATOS_COLUMNAS_BD.get(searchedMetadata.getOid()))) {               
                    sql.append(" AND ");
                    sql.append(this.obtenerOperadorPrepared(searchedMetadata, params, tiposSQL));
                }
            }
            
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarQueryRecursos: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarQueryRecursos: Params: " + params.toString());
            }

			prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();

            while (rs.next()) {
                codigosRecursoList.add(rs.getString("COREC"));
            }
        } catch (NamingException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR, "S12MapasDAO::filtrarQueryRecursos: Exception: " + e.getMessage());
            }
            throw new S12MapasDaoException("S12MapasDAO::filtrarQueryRecursos: Exception ", e);
        } catch (SQLException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR, "S12MapasDAO::filtrarQueryRecursos: Exception: " + e.getMessage());
            }
            throw new S12MapasDaoException("S12MapasDAO::filtrarQueryRecursos: Exception ", e);
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
        } finally {
            S12ConnectionManager.close(rs);
            S12ConnectionManager.close(prepStmt);
            S12ConnectionManager.close(conn);
        }

        String[] codigosRecursoArray = new String[codigosRecursoList.size()];
        codigosRecursoList.toArray(codigosRecursoArray);
        return codigosRecursoArray;
    }

	/**
     * Convierte una operacin de Lote 3 en una operacin de BBDD con PreparedStatement
     *
     * @param operation operacin de Lote 3
     * @param params lista de objetos para valores de sustitucion en condiciones
     * @param tiposSQL lista de tipos SQL de los valores para condiciones
     *
     * @return operacin SQL
     */
    private String obtenerOperadorPrepared(
			R01MSearchedMetaData operation, ArrayList<Object> params,
			ArrayList<Integer> tiposSQL) {
        StringBuffer sqlOperation = new StringBuffer();

        if (S12Constantes.METADATOS_SERVICIO.get(operation.getOid()) != null) {
        	sqlOperation.append(S12Constantes.METADATOS_SERVICIO.get(operation.getOid()));
        	sqlOperation.append(" IS NOT NULL ");       
        } else if (S12MapasDAO.EQ.equals(operation.getOperationSelected())) {
        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_BD.get(operation.getOid()));
            sqlOperation.append(" = ?");
            params.add(operation.getValue());
            tiposSQL.add(Types.VARCHAR);
        } else if (S12MapasDAO.GT.equals(operation.getOperationSelected())) {
        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_BD.get(operation.getOid()));
            sqlOperation.append(" > ?");
            params.add(operation.getValue());
            tiposSQL.add(Types.VARCHAR);
        } else if (S12MapasDAO.GTE.equals(operation.getOperationSelected())) {
        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_BD.get(operation.getOid()));
            sqlOperation.append(" >= ?");
            params.add(operation.getValue());
            tiposSQL.add(Types.VARCHAR);
        } else if (S12MapasDAO.LT.equals(operation.getOperationSelected())) {
        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_BD.get(operation.getOid()));
            sqlOperation.append(" < ?");
            params.add(operation.getValue());
            tiposSQL.add(Types.VARCHAR);
        } else if (S12MapasDAO.LTE.equals(operation.getOperationSelected())) {
        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_BD.get(operation.getOid()));
            sqlOperation.append(" <= ?");
            params.add(operation.getValue());
            tiposSQL.add(Types.VARCHAR);
        } else if (S12MapasDAO.LIKE.equals(operation.getOperationSelected())) {
        	sqlOperation.append(" UPPER(");
        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_BD.get(operation.getOid()));
        	sqlOperation.append(")");
            sqlOperation.append(" LIKE ?");
            String valorLIKE = "%".concat(operation.getValue().toUpperCase(new Locale("es"))).concat("%"); 
            params.add(valorLIKE);
            tiposSQL.add(Types.VARCHAR);
        } else if (S12MapasDAO.IN.equals(operation.getOperationSelected())) {
        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_BD.get(operation.getOid()));
            sqlOperation.append(" IN ");
            sqlOperation.append("(");
            String valor = operation.getValue().replace("(", "").replace(")", "");
            String parametros[] = valor.split(",");
            for (int i = 0; i < parametros.length; i++) {
            	if (i != 0) {
            		sqlOperation.append(",");
            	}
            	sqlOperation.append("?");
                params.add(parametros[i]);
                tiposSQL.add(Types.VARCHAR);
            }
            
            sqlOperation.append(") ");
        }

        return sqlOperation.toString();
	}

    /**
     * Obtiene todos los cdigos de recurso que que cumplen con las condiciones de la
     * consulta pasada por parmetro en formato JSON
     *
     * @param language Idioma
     * @param queryJSONString String JSON con toda la informacin de la consulta
     * 
     * @return Array de String
     * 
     * @throws S12MapasDaoException Excepcin
     */
    public String[] filtrarQueryRecursos(String language, String queryJSONString) throws S12MapasDaoException {
        Connection conn = null;
        PreparedStatement prepStmt = null;
        ResultSet rs = null;
        ArrayList<String> codigosRecursoList = new ArrayList<String>();
	    ArrayList<Object> params = new ArrayList<Object>(); 
	    ArrayList<Integer> tiposSQL = new ArrayList<Integer>();

        try {
            conn = S12ConnectionManager.getConnection();

            if (queryJSONString == null || "".equals(queryJSONString)) {
            	if (Q70Factoria.getTraza().isTraceActive()) {
                    Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarQueryRecursos: queryJSONString: " + queryJSONString);
                }
            	
            	return new String[0];
            }

            StringBuffer sql = new StringBuffer();
            sql.append(" SELECT COREC FROM CONSULTAMAT_RECURSOS_PM WHERE ");
            //si no tiene coordenadas, lo excluimos para el mapa, aunque cumpla los criterios de bsqueda.
            sql.append(" GMLNG IS NOT NULL AND GMLAT IS NOT NULL ");
            sql.append(" AND LANGUAGE = ? ");
            params.add(language);
            tiposSQL.add(Types.VARCHAR);

    		try {
    			JSONObject objetoJSON = new JSONObject(queryJSONString);
    			String[] propiedades = JSONObject.getNames(objetoJSON);
    			for (int i = 0; i < propiedades.length; i++) {
    				String propiedad = propiedades[i];
    	            if (Q70Factoria.getTraza().isTraceActive()) {
    	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarQueryRecursos: (propiedad,valor) = (" + propiedad + " , " + objetoJSON.getString(propiedad) + ")");
    	            }
                    // Comprobamos si el metadato debe ignorarse, si no, lo aadimos a la consulta
                    if (!"".equals(S12Constantes.METADATOS_COLUMNAS_BD.get(propiedades[i]))) {
                    	String condicionSQL = this.obtenerOperadorPrepared(propiedad, objetoJSON.getString(propiedad), params, tiposSQL);
                    	if (!"".equals(condicionSQL)){
	                        sql.append(" AND ");
	                        sql.append(condicionSQL);
                    	}
                    }
    			}
    		} catch (JSONException e1) {
                if (Q70Factoria.getTraza().isTraceActive()) {
                    Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR, "S12MapasDAO::filtrarQueryRecursos: Exception: " + e1.getMessage());
                }
    		}
            
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarQueryRecursos: Query: " + sql.toString());
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::filtrarQueryRecursos: Params: " + params.toString());
            }
			
    		prepStmt = conn.prepareStatement(sql.toString());
			int index = 1;
			for (int i = 0; i < params.size(); i++) {
				prepStmt.setObject(index++, params.get(i), tiposSQL.get(i));	
			}
			
			rs = prepStmt.executeQuery();

            while (rs.next()) {
                codigosRecursoList.add(rs.getString("COREC"));
            }
        } catch (NamingException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR, "S12MapasDAO::filtrarQueryRecursos: Exception: " + e.getMessage());
            }
            throw new S12MapasDaoException("S12MapasDAO::filtrarQueryRecursos: Exception ", e);
        } catch (SQLException e) {
            if (Q70Factoria.getTraza().isTraceActive()) {
                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR, "S12MapasDAO::filtrarQueryRecursos: Exception: " + e.getMessage());
            }
            throw new S12MapasDaoException("S12MapasDAO::filtrarQueryRecursos: Exception ", e);
	    } catch (NullPointerException e) {
	        if (Q70Factoria.getTraza().isTraceActive()) {
	                Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.ERROR,"S12MapasDAO::filtrarRecursos: Exception: " + e.getMessage());
	            }
	      throw new S12MapasDaoException("S12MapasDAO::filtrarRecursos: Exception ", e );
        } finally {
            S12ConnectionManager.close(rs);
            S12ConnectionManager.close(prepStmt);
            S12ConnectionManager.close(conn);
        }

        String[] codigosRecursoArray = new String[codigosRecursoList.size()];
        codigosRecursoList.toArray(codigosRecursoArray);
        return codigosRecursoArray;
    }

	/**
     * Adaptacin del mtodo que convierte una operacin de Lote 3 en una 
     * operacin de BBDD. En esta adaptacin nos llega el nombre del
     * metadato con un valor asociado, pero sin operador explcito, que
     * en este caso es "calculado" dentro del mtodo.
     *
     * @param nombrePropiedad nombre del metadato
     * @param valorPropiedad valor para la bsqueda
     * @param params lista de objetos para valores de sustitucion en condiciones
     * @param tiposSQL lista de tipos SQL de los valores para condiciones
     *
     * @return operacin SQL
     */
    private String obtenerOperadorPrepared(String nombreMetadato, String valorMetadato,
    		ArrayList<Object> params, ArrayList<Integer> tiposSQL) {
        StringBuffer sqlOperation = new StringBuffer();

        //Primero: Caso particular para arreglar problema en buscador B1, ignoramos valores de restorationType que en realidad son porque comparten combo con K8 y L2 (zonas y bares de pintxos)
        if (nombreMetadato.equals(S12Constantes.SF_RESTORATION_TYPE) && 
        		(valorMetadato.equals("0002") || valorMetadato.equals("0004"))) {
        	return "";
        }

        //Para deducir el operador a utilizar, se han definido unos HashMap para cada tipo, 
        //sobre los cuales comprobamos la pertenendia del metadato recibido.
        if (S12Constantes.METADATOS_SERVICIO.get(nombreMetadato) != null) {
        	sqlOperation.append(S12Constantes.METADATOS_SERVICIO.get(nombreMetadato));
        	sqlOperation.append(" IS NOT NULL ");       
        } else if (S12Constantes.METADATOS_COLUMNAS_IN_BD.get(nombreMetadato) != null) {
        	//No se incluye condicin cuando el valor es vaco
        	if (!"".equals(valorMetadato)){
	        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_IN_BD.get(nombreMetadato));
	            sqlOperation.append(" IN (");
	            String valor = valorMetadato.replace("(", "").replace(")", "");
	            String parametros[] = valor.split(",");
	            for (int i = 0; i < parametros.length; i++) {
	            	if (i != 0) {
	            		sqlOperation.append(",");
	            	}
	            	sqlOperation.append("?");
	            	params.add(parametros[i]);
	            	tiposSQL.add(Types.VARCHAR);
	            }
	            sqlOperation.append(") ");
        	}
        } else if (S12Constantes.METADATOS_COLUMNAS_EQ_BD.get(nombreMetadato) != null) {
        	//No se incluye condicin cuando el valor es vaco
        	if (!"".equals(valorMetadato)){
	        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_EQ_BD.get(nombreMetadato));
	            sqlOperation.append(" = ?");
            	params.add(valorMetadato);
            	tiposSQL.add(Types.VARCHAR);
        	}
        } else if (S12Constantes.METADATOS_COLUMNAS_LIKE_BD.get(nombreMetadato) != null) {
        	//No se incluye condicin cuando el valor es vaco
        	if (!"".equals(valorMetadato)){
	        	sqlOperation.append(" UPPER(");
	        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_LIKE_BD.get(nombreMetadato));
	        	sqlOperation.append(")");
	        	sqlOperation.append(" LIKE ?");           
	        	String valorLIKE = S12Utils.utf8toIso88591(valorMetadato);
	        	valorLIKE = valorLIKE.toUpperCase(new Locale("es", "es"));
	        	valorLIKE = "%".concat(valorLIKE).concat("%");
            	params.add(valorLIKE);
            	tiposSQL.add(Types.VARCHAR);
        	}
        } else if (S12Constantes.METADATOS_COLUMNAS_GTE_BD.get(nombreMetadato) != null) {
        	//No se incluye condicin cuando el valor es vaco
        	if (!"".equals(valorMetadato)){
	        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_GTE_BD.get(nombreMetadato));
	            sqlOperation.append(" >= ?");
            	params.add(valorMetadato);
            	tiposSQL.add(Types.VARCHAR);
        	}
        } else if (S12Constantes.METADATOS_COLUMNAS_LTE_BD.get(nombreMetadato) != null) {
        	//No se incluye condicin cuando el valor es vaco
        	if (!"".equals(valorMetadato)){
	        	sqlOperation.append(S12Constantes.METADATOS_COLUMNAS_LTE_BD.get(nombreMetadato));
	            sqlOperation.append(" <= ?");
            	params.add(valorMetadato);
            	tiposSQL.add(Types.VARCHAR);
        	}
        }        

        if (Q70Factoria.getTraza().isTraceActive()) {
            Q70Factoria.getTraza().trace(S12MapasDAO.COMPONENTE, Q70TraceLevel.DEBUG, "S12MapasDAO::obtenerOperador("+nombreMetadato+","+valorMetadato+") = " + sqlOperation.toString());
        }
        
        return sqlOperation.toString();
    }

    /**
     * Utilidad para tratar listas de valores usadas en clausulas IN() dentro de MapasDAO
     * (solo est preparada para tratar VARCHARs o NUMERIC sin decimales)
     * 
     * @param listaValores
     * @param params lista de objetos para valores de sustitucion en condiciones
     * @param tiposSQL lista de tipos SQL de los valores para condiciones
     * @return string con la lista de parametros (ejemplo: "?,?,?,?")
     */
    private String parametrizarValoresIN(String listaValores, ArrayList<Object> params, ArrayList<Integer> tiposSQL) {
    	StringBuffer listaParams = new StringBuffer();
    	
    	String[] arrValores = listaValores.split(",");
    	for (int i = 0; i < arrValores.length; i++) {
    		String valor = arrValores[i].trim();
			//pueden ser numeros o cadenas entre comillas:
    		if ("'".equals(valor.substring(0, 1))){
    			//parametrizar un VARCHAR, quitando las comillas simples
    			valor = valor.substring(1, valor.length()-1);
    			params.add(valor);
    			tiposSQL.add(Types.VARCHAR);
    		} else {
    			//parametrizar un LONG
    			params.add(new Long(valor));
    			tiposSQL.add(Types.NUMERIC);
    		}
    		if(i!=0){
    			listaParams.append(",");
    		}
			listaParams.append("?");
		}
    	return listaParams.toString();
    }
}
