package s12.buscadorText.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.Normalizer;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.naming.NamingException;

import s12.buscadorText.valueObjects.S12EnlaceMunicipio;
import s12.buscadorText.valueObjects.S12RecursoBuscadorText;
import s12.constantes.S12Constantes;
import s12.util.S12ConnectionManager;
import s12.util.S12Utils;
import bilbofrmk.util.BLogger;
import es.ejie.frmk.Q70Factoria;
import es.ejie.frmk.infraestructura.internacionalizacion.Q70GestorMensajesException;
import es.ejie.frmk.infraestructura.internacionalizacion.Q70GestorMensajesi18n;
import es.ejie.frmk.infraestructura.traza.Q70TraceLevel;
import es.ejie.frmk.listeners.base.Q70ListenerUtils;

/**
 * 
 *  
 */
public class S12BuscadorTextDAO {

	/**
	 * Constante que indica el componente al que pertenece dicha clase. Es necesario para el uso del log de trazas.
	 */
	private static final String COMPONENTE = "BUSCADOR_TEXT";

	/** The Constant SIZE_STRINGBUFFER. */
	private static final int SIZE_STRINGBUFFER = 220;


	/**
	 * getRecursosOracleText devuelve los recursos que cumplen las condiciones
	 * @param texto
	 * @param idioma
	 * @param numero_resultados
	 * @param ultimo_resultado
	 * @return
	 * @throws S12BuscadorTextDaoException
	 */
	public ArrayList<S12RecursoBuscadorText> getRecursosOracleText(String texto, String idioma,
			int numero_resultados, int ultimo_resultado) throws S12BuscadorTextDaoException {
		Connection conn = null;
		Statement stmt = null;		
		PreparedStatement prepStmt = null;
		ResultSet rs = null;
		ArrayList<S12RecursoBuscadorText> listaRecursos = new ArrayList<S12RecursoBuscadorText>();
		
		StringBuffer sql;
		
		//le quitamos tildes y lo ponemos en mayusculas para la busqueda de localidad
		String textoLocalidades = "";
		textoLocalidades = stripAccents(texto);
		textoLocalidades = textoLocalidades.toUpperCase();

		try {

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"getRecursosOracleText: texto recibido: " + texto);
			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"getRecursosOracleText: idioma recibido: " + idioma);
			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"getRecursosOracleText: numero_resultados recibido: " + numero_resultados);
			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"getRecursosOracleText: ultimo_resultado recibido: " + ultimo_resultado);

			conn = S12ConnectionManager.getConnection();
			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("alter session set nls_comp=linguistic");
			stmt = conn.createStatement();
			stmt.execute(sql.toString());

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);
			sql.append("alter session set nls_sort=binary_ai");
			stmt.execute(sql.toString());

			int numFilaMax = ultimo_resultado + numero_resultados;

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("select * from ( select a.*, rownum rnum from (");
			
			sql.append("SELECT DISTINCT(COREC) COREC, ORDEN, NOMBRE, SCORE(1) a, SCORE(2) b ");
			sql.append(" FROM CONSULTAMAT_BUSCADOR WHERE ");
			sql.append(" COIDIOMA = ? ");
			sql.append(" AND COTIPREC <> ? ");
			sql.append(" AND (CONTAINS(NOMBRE, ?, 1) > 0 OR CONTAINS(DESCRIPCION, ?, 2) > 0 ) " );		
			sql.append(" UNION ");
			sql.append("SELECT DISTINCT(COREC) COREC, ORDEN, NOMBRE, 0 a, 0 b ");
			sql.append(" FROM CONSULTAMAT_BUSCADOR WHERE ");
			sql.append(" COIDIOMA = ? ");
			sql.append(" AND COTIPREC = ? ");
			//sql.append(" AND (CONTAINS(NOMBRE, ?, 1) > 0 OR CONTAINS(DESCLOCALIDAD, ?) > 0 ) " );					
			sql.append(" AND (TRANSLATE(UPPER(NOMBRE),'', 'aeiouAEIOU') LIKE ?  ");
			sql.append(" OR TRANSLATE(UPPER(DESCLOCALIDAD),'', 'aeiouAEIOU') LIKE ? )  ");
			sql.append(" GROUP BY COREC, ORDEN, NOMBRE ");
			sql.append(" ORDER BY ORDEN, a desc, b desc, NOMBRE ");
		
			sql.append(") a where rownum <= " + numFilaMax + " ) where rnum > " + ultimo_resultado);

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"S12BuscadorTextDAO::getRecursosOracleText: SQL = " + sql.toString());

			prepStmt = conn.prepareStatement(sql.toString());
			
			int index = 1;
			prepStmt.setString(index++, idioma);
			prepStmt.setString(index++, S12Constantes.LOCALIDAD);
			prepStmt.setString(index++, "%" + texto + "%");
			prepStmt.setString(index++, "%" + texto + "%");
			prepStmt.setString(index++, idioma);
			prepStmt.setString(index++, S12Constantes.LOCALIDAD);
			prepStmt.setString(index++, "%" + textoLocalidades + "%");
			prepStmt.setString(index++, "%" + textoLocalidades + "%");			
			
			rs = prepStmt.executeQuery();			
			
			while (rs.next()) {
				S12RecursoBuscadorText recurso = new S12RecursoBuscadorText();
				recurso.setCodigoRecurso(rs.getLong("COREC"));				
				listaRecursos.add(recurso);
			}
		} catch (NamingException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::getRecursosOracleText: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::getRecursosOracleText: Exception ", e);
		} catch (SQLException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::getRecursosOracleText: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::getRecursosOracleText: Exception ", e);
		} finally {
			S12ConnectionManager.close(rs);
			S12ConnectionManager.close(stmt);
			S12ConnectionManager.close(conn);
		}

		return listaRecursos;
	}
	
	public static String stripAccents(String str) {
		String ORIGINAL = "";
		String REPLACEMENT = "AaEeIiOoUu";
	    if (str == null) {
	        return null;
	    }
	    char[] array = str.toCharArray();
	    for (int index = 0; index < array.length; index++) {
	        int pos = ORIGINAL.indexOf(array[index]);
	        if (pos > -1) {
	            array[index] = REPLACEMENT.charAt(pos);
	        }
	    }
	    return new String(array);
	}
	
	/**
	 * findTipoPlantillaRuta devuelve el tipo de plantilla del recurso si es una RUTA
	 * @param codigoRecurso
	 * @param idioma
	 * @return
	 * @throws S12BuscadorTextDaoException
	 */
	public String findTipoPlantillaRuta(Long codigoRecurso, String idioma) throws S12BuscadorTextDaoException {
		StringBuilder query = new StringBuilder();
		Connection conn = null;
		Statement stmt = null;
		PreparedStatement prepStmt = null;		
		ResultSet rs = null;
		StringBuffer sql;
		String tipoPlantilla = "";
		try {

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"findTipoPlantillaRuta: codigoRecurso recibido: " + codigoRecurso);
			

			conn = S12ConnectionManager.getConnection();
			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("alter session set nls_comp=linguistic");
			stmt = conn.createStatement();
			stmt.execute(sql.toString());

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);
			sql.append("alter session set nls_sort=binary_ai");
			stmt.execute(sql.toString());

		

		query.append("SELECT TIPOPLANTILLA FROM RUTAS WHERE COREC = ? AND COIDIOMA = ? ");
		
		S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
				"S12BuscadorTextDAO::findTipoPlantillaRuta: SQL = " + query.toString());

		prepStmt = conn.prepareStatement(query.toString());
		
		int index = 1;
		
		prepStmt.setLong(index++, codigoRecurso);	
		prepStmt.setString(index++, idioma);	
		
		rs = prepStmt.executeQuery();
		

		while (rs.next()) {
			tipoPlantilla = rs.getString("TIPOPLANTILLA");
		}
		} catch (NamingException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findTipoPlantillaRuta: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findTipoPlantillaRuta: Exception ", e);
		} catch (SQLException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findTipoPlantillaRuta: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findTipoPlantillaRuta: Exception ", e);
		} finally {
			S12ConnectionManager.close(rs);
			S12ConnectionManager.close(stmt);
			S12ConnectionManager.close(conn);
		}
			return tipoPlantilla;
		
	}
	/**
	 * findTipoPlantillaEtapa devuelve el tipo de plantilla del recurso si es una ETAPA
	 * @param codigoRecurso
	 * @param idioma
	 * @return
	 * @throws S12BuscadorTextDaoException
	 */
	public String findTipoPlantillaEtapa(Long codigoRecurso, String idioma) throws S12BuscadorTextDaoException {
		StringBuilder query = new StringBuilder();
		Connection conn = null;
		Statement stmt = null;
		PreparedStatement prepStmt = null;
		ResultSet rs = null;
		StringBuffer sql;
		String tipoPlantilla = "";
		try {

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"findTipoPlantillaEtapa: codigoRecurso recibido: " + codigoRecurso);
			

			conn = S12ConnectionManager.getConnection();
			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("alter session set nls_comp=linguistic");
			stmt = conn.createStatement();
			stmt.execute(sql.toString());

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);
			sql.append("alter session set nls_sort=binary_ai");
			stmt.execute(sql.toString());

		

		query.append("SELECT TIPOPLANTILLA FROM ETAPAS WHERE COREC = ? AND COIDIOMA = ? ");
		
		S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
				"S12BuscadorTextDAO::findTipoPlantillaEtapa: SQL = " + query.toString());

prepStmt = conn.prepareStatement(query.toString());
		
		int index = 1;
		
		prepStmt.setLong(index++, codigoRecurso);	
		prepStmt.setString(index++, idioma);	
		
		
		rs = prepStmt.executeQuery();
		

		while (rs.next()) {
			tipoPlantilla = rs.getString("TIPOPLANTILLA");
		}
		} catch (NamingException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findTipoPlantillaEtapa: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findTipoPlantillaEtapa: Exception ", e);
		} catch (SQLException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findTipoPlantillaEtapa: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findTipoPlantillaEtapa: Exception ", e);
		} finally {
			S12ConnectionManager.close(rs);
			S12ConnectionManager.close(stmt);
			S12ConnectionManager.close(conn);
		}
			return tipoPlantilla;
		
	}
	
	/**
	 * findUrlAmigable devuelve la url amigable del recurso
	 * @param codigoRecurso
	 * @param idioma
	 * @return
	 * @throws S12BuscadorTextDaoException
	 */
	public String findUrlAmigable(Long codigoRecurso, String idioma) throws S12BuscadorTextDaoException {
		StringBuilder query = new StringBuilder();
		Connection conn = null;
		Statement stmt = null;
		PreparedStatement prepStmt = null;
		ResultSet rs = null;
		StringBuffer sql;
		String urlAmigable = "";
		try {

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"findUrlAmigable: codigoRecurso recibido: " + codigoRecurso);
			

			conn = S12ConnectionManager.getConnection();
			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("alter session set nls_comp=linguistic");
			stmt = conn.createStatement();
			stmt.execute(sql.toString());

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);
			sql.append("alter session set nls_sort=binary_ai");
			stmt.execute(sql.toString());

		

		query.append("SELECT URLAMIGABLE FROM POSICIONAMIENTO WHERE COREC = ? AND COIDIOMA = ? ");
		
		S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
				"S12BuscadorTextDAO::findUrlAmigable: SQL = " + query.toString());

prepStmt = conn.prepareStatement(query.toString());
		
		int index = 1;
		
		prepStmt.setLong(index++, codigoRecurso);	
		prepStmt.setString(index++, idioma);	
		
		
		rs = prepStmt.executeQuery();
		

		while (rs.next()) {
			urlAmigable = rs.getString("URLAMIGABLE");
		}
		} catch (NamingException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findUrlAmigable: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findUrlAmigable: Exception ", e);
		} catch (SQLException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findUrlAmigable: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findUrlAmigable: Exception ", e);
		} finally {
			S12ConnectionManager.close(rs);
			S12ConnectionManager.close(stmt);
			S12ConnectionManager.close(conn);
		}
			return urlAmigable;
		
	}
	
	/**
	 * findRutaEtapa devuelve la primera ruta que encuentra asociada a una etapa
	 * @param codigoRecurso de la etapa
	 * @return
	 * @throws S12BuscadorTextDaoException
	 */
	public Long findRutaEtapa(Long codigoRecurso) throws S12BuscadorTextDaoException {
		StringBuilder query = new StringBuilder();
		Connection conn = null;
		Statement stmt = null;
		PreparedStatement prepStmt = null;
		ResultSet rs = null;
		StringBuffer sql;
		Long rutaEtapa = new Long("0");
		try {

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"findRutaEtapa: codigoRecurso recibido: " + codigoRecurso);
			

			conn = S12ConnectionManager.getConnection();
			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("alter session set nls_comp=linguistic");
			stmt = conn.createStatement();
			stmt.execute(sql.toString());

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);
			sql.append("alter session set nls_sort=binary_ai");
			stmt.execute(sql.toString());

		

		query.append("SELECT CORUTA FROM RUTAS_ETAPAS WHERE COETAPA = ? " );
		
		S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
				"S12BuscadorTextDAO::findRutaEtapa: SQL = " + query.toString());

prepStmt = conn.prepareStatement(query.toString());
		
		int index = 1;
		
		prepStmt.setLong(index++, codigoRecurso);	
		
		
		rs = prepStmt.executeQuery();
		

		while (rs.next()) {
			rutaEtapa = rs.getLong("CORUTA");
		}
		} catch (NamingException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findRutaEtapa: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findRutaEtapa: Exception ", e);
		} catch (SQLException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
						"S12BuscadorTextDAO::findRutaEtapa: Exception: " + e.getMessage());
			}
			throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findRutaEtapa: Exception ", e);
		} finally {
			S12ConnectionManager.close(rs);
			S12ConnectionManager.close(stmt);
			S12ConnectionManager.close(conn);
		}
			return rutaEtapa;
		
	}
	
	/**
	 * findEnlacesMunicipio devuelve la lista de municipios con sus urls
	 * @param codigoRecurso
	 * @param idioma
	 * @return
	 * @throws S12BuscadorTextDaoException
	 */
	public List<S12EnlaceMunicipio> findEnlacesMunicipio(Long codigoRecurso, String idioma) throws S12BuscadorTextDaoException {
		StringBuilder query = new StringBuilder();
		Connection conn = null;
		Statement stmt = null;
		PreparedStatement prepStmt = null;
		ResultSet rs = null;
		StringBuffer sql;
		ArrayList<S12EnlaceMunicipio> listaEnlaces = new ArrayList<S12EnlaceMunicipio>();
		try {

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"findEnlacesMunicipio: codigoRecurso recibido: " + codigoRecurso);
			

			conn = S12ConnectionManager.getConnection();
			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("alter session set nls_comp=linguistic");
			stmt = conn.createStatement();
			stmt.execute(sql.toString());

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);
			sql.append("alter session set nls_sort=binary_ai");
			stmt.execute(sql.toString());

		

		query.append("SELECT DISTINCT RECURSO.COREC CODIGO_RECURSO, T17_MUNICIPIO.DS_O NOMBRE_MUNICIPIO, T17_PROVINCIA.DS_O NOMBRE_PROVINCIA, ");
		query.append(" POSICIONAMIENTO_ES.URLAMIGABLE URLAMIGABLE_ES, POSICIONAMIENTO_EU.URLAMIGABLE URLAMIGABLE_EU, ");
		query.append(" POSICIONAMIENTO_EN.URLAMIGABLE URLAMIGABLE_EN, POSICIONAMIENTO_FR.URLAMIGABLE URLAMIGABLE_FR, ");
		query.append(" POSICIONAMIENTO_DE.URLAMIGABLE URLAMIGABLE_DE ");
		query.append(" FROM RECURSO ");
		query.append(" INNER JOIN REC_GEOCATALOG RECGEO ON RECURSO.COREC = RECGEO.COREC ");
		query.append(
				" INNER JOIN T17_MUNICIPIO ON RECGEO.COMUN   = T17_MUNICIPIO.ID AND RECGEO.COPROV = T17_MUNICIPIO.PROVINCIA_ID ");
		query.append(
				" INNER JOIN T17_PROVINCIA ON RECGEO.COPROV   = T17_PROVINCIA.ID  ");
		query.append(
				" LEFT JOIN POSICIONAMIENTO POSICIONAMIENTO_ES ON RECURSO.COREC = POSICIONAMIENTO_ES.COREC AND POSICIONAMIENTO_ES.COIDIOMA = ? ");
		query.append(
				" LEFT JOIN POSICIONAMIENTO POSICIONAMIENTO_EU ON RECURSO.COREC = POSICIONAMIENTO_EU.COREC AND POSICIONAMIENTO_EU.COIDIOMA = ? ");
		query.append(
				" LEFT JOIN POSICIONAMIENTO POSICIONAMIENTO_EN ON RECURSO.COREC = POSICIONAMIENTO_EN.COREC AND POSICIONAMIENTO_EN.COIDIOMA = ? ");
		query.append(
				" LEFT JOIN POSICIONAMIENTO POSICIONAMIENTO_FR ON RECURSO.COREC = POSICIONAMIENTO_FR.COREC AND POSICIONAMIENTO_FR.COIDIOMA = ? ");
		query.append(
				" LEFT JOIN POSICIONAMIENTO POSICIONAMIENTO_DE ON RECURSO.COREC = POSICIONAMIENTO_DE.COREC AND POSICIONAMIENTO_DE.COIDIOMA = ? ");
		query.append(" WHERE RECURSO.COTIPREC = ? AND ");
		query.append("  EXISTS ");
		query.append(" ( ");
		query.append(" 		SELECT 1 FROM REC_GEOCATALOG RECGEOMUN ");
		query.append(" 		WHERE RECGEOMUN.COPROV = RECGEO.COPROV ");
		query.append(" 		AND RECGEOMUN.COMUN = RECGEO.COMUN ");
		query.append(" 		AND RECGEOMUN.COREC = ? ");
		query.append("  ) ");
		query.append(" AND EXISTS ");
		query.append(" ( ");
		query.append(" 		SELECT 1 FROM REC_GEOCATALOG RECGEOPROV ");
		query.append(" 		WHERE RECGEOPROV.COPROV = RECGEO.COPROV ");
		query.append(" 		AND RECGEOPROV.COREC =   ? ");
		query.append("  ) ");
		query.append(" ORDER BY T17_MUNICIPIO.DS_O ");
		
		S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
				"S12BuscadorTextDAO::findEnlacesMunicipio: SQL = " + query.toString());

prepStmt = conn.prepareStatement(query.toString());
		
		int index = 1;
		
		prepStmt.setString(index++, S12Constantes.IDIOMA_ES);	
		prepStmt.setString(index++, S12Constantes.IDIOMA_EU);	
		prepStmt.setString(index++, S12Constantes.IDIOMA_EN);	
		prepStmt.setString(index++, S12Constantes.IDIOMA_FR);			
		prepStmt.setString(index++, S12Constantes.IDIOMA_DE);		
		prepStmt.setString(index++, S12Constantes.LOCALIDAD);			
		prepStmt.setLong(index++, codigoRecurso);	
		prepStmt.setLong(index++, codigoRecurso);	
		
		
		rs = prepStmt.executeQuery();
		

		while (rs.next()) {
			S12EnlaceMunicipio enlace = new S12EnlaceMunicipio();

			enlace.setCodigoRecurso(rs.getLong("CODIGO_RECURSO"));
			enlace.setNombreMunicipio(rs.getString("NOMBRE_MUNICIPIO"));
			enlace.setNombreProvincia(rs.getString("NOMBRE_PROVINCIA"));
			if(idioma.equalsIgnoreCase(S12Constantes.IDIOMA_ES)){
				enlace.setUrlAmigable(rs.getString("URLAMIGABLE_ES"));				
			}
			if(idioma.equalsIgnoreCase(S12Constantes.IDIOMA_EU)){
				enlace.setUrlAmigable(rs.getString("URLAMIGABLE_EU"));				
			}
			if(idioma.equalsIgnoreCase(S12Constantes.IDIOMA_EN)){
				enlace.setUrlAmigable(rs.getString("URLAMIGABLE_EN"));				
			}
			if(idioma.equalsIgnoreCase(S12Constantes.IDIOMA_FR)){
				enlace.setUrlAmigable(rs.getString("URLAMIGABLE_FR"));				
			}
			if(idioma.equalsIgnoreCase(S12Constantes.IDIOMA_DE)){
				enlace.setUrlAmigable(rs.getString("URLAMIGABLE_DE"));				
			}
			

			listaEnlaces.add(enlace);
			
		}
	} catch (NamingException e) {
		if (Q70Factoria.getTraza().isTraceActive()) {
			Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
					"S12BuscadorTextDAO::findEnlacesMunicipio: Exception: " + e.getMessage());
		}
		throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findEnlacesMunicipio: Exception ", e);
	} catch (SQLException e) {
		if (Q70Factoria.getTraza().isTraceActive()) {
			Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
					"S12BuscadorTextDAO::findEnlacesMunicipio: Exception: " + e.getMessage());
		}
		throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::getRecursosOracleText: Exception ", e);
	} finally {
		S12ConnectionManager.close(rs);
		S12ConnectionManager.close(stmt);
		S12ConnectionManager.close(conn);
	}
		return listaEnlaces;

	}
	
	/**
	 * findDatosRecurso devuelve el resto de datos de la vita
	 * @param codigoRecurso
	 * @param idioma
	 * @return
	 * @throws S12BuscadorTextDaoException
	 */
	public S12RecursoBuscadorText findDatosRecurso(Long codigoRecurso, String idioma) throws S12BuscadorTextDaoException {
		StringBuilder query = new StringBuilder();
		Connection conn = null;
		Statement stmt = null;
		PreparedStatement prepStmt = null;
		ResultSet rs = null;
		StringBuffer sql;
		S12RecursoBuscadorText recurso = new S12RecursoBuscadorText();
		try {

			S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
					"findDatosRecurso: codigoRecurso recibido: " + codigoRecurso);
			

			conn = S12ConnectionManager.getConnection();
			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);

			sql.append("alter session set nls_comp=linguistic");
			stmt = conn.createStatement();
			stmt.execute(sql.toString());

			sql = new StringBuffer(S12BuscadorTextDAO.SIZE_STRINGBUFFER);
			sql.append("alter session set nls_sort=binary_ai");
			stmt.execute(sql.toString());

		

		query.append("SELECT COREC, NOMBRE, DESCRIPCION, TELEFONO1, FECHAINICIO, FECHAFIN, COTIPREC, FECHA ");
		query.append(" FROM CONSULTAMAT_BUSCADOR WHERE ");
		query.append(" COIDIOMA = ? AND COREC = ? ");
		
		S12Utils.trazaLarga(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.DEBUG,
				"S12BuscadorTextDAO::findDatosRecurso: SQL = " + query.toString());

		
prepStmt = conn.prepareStatement(query.toString());
		
		int index = 1;
		
		prepStmt.setString(index++, idioma);	
		prepStmt.setLong(index++, codigoRecurso);	
		
		
		rs = prepStmt.executeQuery();
		

		while (rs.next()) {
			recurso.setCodigoRecurso(rs.getLong("COREC"));
			recurso.setNombre(rs.getString("NOMBRE"));
			recurso.setDescripcion(rs.getString("DESCRIPCION"));
			recurso.setTelefono(rs.getString("TELEFONO1"));		
			if(rs.getDate("FECHAINICIO")!=null){
			recurso.setFechaInicio(S12Utils.getFechaFormateada(rs.getDate("FECHAINICIO"), idioma));
			}else{
				recurso.setFechaInicio("");
			}
			if(rs.getDate("FECHAFIN")!=null){
			recurso.setFechaFin(S12Utils.getFechaFormateada(rs.getDate("FECHAFIN"), idioma));
			}else{
				recurso.setFechaFin("");
			}
			if(rs.getDate("FECHA")!=null){
				recurso.setFechaNoticia(S12Utils.getFechaFormateada(rs.getDate("FECHA"), idioma));
				}else{
					recurso.setFechaNoticia("");
				}
			
			recurso.setCodigoTipologia(rs.getString("COTIPREC"));		

			
		}
	} catch (NamingException e) {
		if (Q70Factoria.getTraza().isTraceActive()) {
			Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
					"S12BuscadorTextDAO::findDatosRecurso: Exception: " + e.getMessage());
		}
		throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findDatosRecurso: Exception ", e);
	} catch (SQLException e) {
		if (Q70Factoria.getTraza().isTraceActive()) {
			Q70Factoria.getTraza().trace(S12BuscadorTextDAO.COMPONENTE, Q70TraceLevel.ERROR,
					"S12BuscadorTextDAO::getRecursosOracleText: Exception: " + e.getMessage());
		}
		throw new S12BuscadorTextDaoException("S12BuscadorTextDAO::findDatosRecurso: Exception ", e);
	} finally {
		S12ConnectionManager.close(rs);
		S12ConnectionManager.close(stmt);
		S12ConnectionManager.close(conn);
	}
		return recurso;

	}
	
	
}
