package aa14f.client.api;

import java.util.Collection;

import org.joda.time.LocalDate;

import com.google.common.collect.Lists;
import com.google.inject.Singleton;

import aa14f.model.AA14OrgDivision;
import aa14f.model.AA14OrgDivisionService;
import aa14f.model.AA14OrgDivisionServiceLocation;
import aa14f.model.AA14Organization;
import aa14f.model.AA14Schedule;
import aa14f.model.AA14ScheduleBookingConfig;
import aa14f.model.oids.AA14IDs.AA14OrgDivisionID;
import aa14f.model.oids.AA14IDs.AA14OrgDivisionServiceID;
import aa14f.model.oids.AA14IDs.AA14OrgDivisionServiceLocationID;
import aa14f.model.oids.AA14IDs.AA14ScheduleID;
import aa14f.model.oids.AA14OIDs.AA14OrgDivisionOID;
import aa14f.model.oids.AA14OIDs.AA14OrgDivisionServiceLocationOID;
import aa14f.model.oids.AA14OIDs.AA14OrgDivisionServiceOID;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import r01f.locale.Language;
import r01f.types.Color;
import r01f.types.contact.ContactInfo;
import r01f.types.datetime.Time;
import r01f.types.geo.GeoOIDs.GeoZipCode;
import r01f.types.geo.GeoPosition;
import r01f.types.geo.GeoStreet;

@Slf4j
@Singleton
@Accessors(prefix="_")
public class AA14ConfigForMedicalService 
	 extends AA14ConfigForEJGVBase {
/////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTS
/////////////////////////////////////////////////////////////////////////////////////////
	// division & service
	public static final AA14OrgDivisionID MEDICAL_ID = AA14OrgDivisionID.forId("EJGV_MEDICO");
	public static final AA14OrgDivisionServiceID SERVICE_ID = AA14OrgDivisionServiceID.forId("SERVICIO_MEDICO");
	
	// locations & schedules
	public static final AA14OrgDivisionServiceLocationID LOC_SERVICIO_MEDICO_AR_ID = AA14OrgDivisionServiceLocationID.forId("SERVICIO_MEDICO_AR");
	public static final AA14ScheduleID SCH_SERVICIO_MEDICO_AR_ID = AA14ScheduleID.forId("SCH_SERVICIO_MEDICO_AR");
		
	public static final AA14OrgDivisionServiceLocationID LOC_SERVICIO_MEDICO_GI_ID = AA14OrgDivisionServiceLocationID.forId("SERVICIO MEDICO_GI");
	public static final AA14ScheduleID SCH_SERVICIO_MEDICO_GI_ID = AA14ScheduleID.forId("SCH_SERVICIO_MEDICO_GI");
	
	public static final AA14OrgDivisionServiceLocationID LOC_SERVICIO_MEDICO_BIZ_ID = AA14OrgDivisionServiceLocationID.forId("SERVICIO MEDICO_BIZ");
	public static final AA14ScheduleID SCH_SERVICIO_MEDICO_BIZ_ID = AA14ScheduleID.forId("SCH_SERVICIO_MEDICO_BIZ");
	
	// colors
	private static final Color COLOR_SERVICIO_MEDICO = Color.from("LightGreen");
	
	// Texts
	private static final String SERVICIO_MEDICO_TXT_ES = "Servicio mdico";
	private static final String SERVICIO_MEDICO_TXT_EU = "Zerbitzu medikoa";
	
	
/////////////////////////////////////////////////////////////////////////////////////////
//  
/////////////////////////////////////////////////////////////////////////////////////////
	public AA14OrgDivisionService getDonacionSangreService() {
		return this.getServiceFor(SERVICE_ID);
	}
	public AA14OrgDivisionServiceLocation getDonacionSangreArabaLocation() {
		return this.getLocationFor(LOC_SERVICIO_MEDICO_AR_ID);
	}
	public AA14OrgDivisionServiceLocation getDonacionSangreGipuzkoaLocation() {
		return this.getLocationFor(LOC_SERVICIO_MEDICO_GI_ID);
	}
	public AA14OrgDivisionServiceLocation getDonacionSangreBizkaiaLocation() {
		return this.getLocationFor(LOC_SERVICIO_MEDICO_BIZ_ID);
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  METHODS
/////////////////////////////////////////////////////////////////////////////////////////
	@Override
	protected AA14OrgDivision _loadOrCreateOrgDivisionConfig(final AA14Organization org) {
		// --- Division
		AA14OrgDivision division = _clientApi.orgDivisionsAPI()
											 .getForCRUD()
											 .loadByIdOrNull(MEDICAL_ID);
		if (division == null) {
			log.warn("\t[Division]: SERVICIO_MEDICO did NOT previously exists... creating it");
			division = _buildDivision(org,
									  AA14OrgDivisionOID.supply(),MEDICAL_ID,
									  "Servicio mdico","Zerbitzu medikoa");
			division = _clientApi.orgDivisionsAPI()
									  .getForCRUD()
									  .save(division);
		}
		return division;
	}
	@Override
	protected Collection<AA14OrgDivisionService> _loadOrCreateOrgDivisionServicesConfig(final AA14Organization org,
									  							      			  		final AA14OrgDivision division) {
		Collection<AA14OrgDivisionService> outServices = Lists.newArrayListWithExpectedSize(3);
		// --- Service: Donacion de sangre
		{
			AA14OrgDivisionService bloodDonationService = _clientApi.orgDivisionServicesAPI()
																		  .getForCRUD()
																		  .loadByIdOrNull(SERVICE_ID);
			if (bloodDonationService == null) {
				log.warn("\t\t[Service]: SERVICIO MEDICO. NOT previously exists... creating it");
				bloodDonationService = _buildService(org,
												     division,
													 AA14OrgDivisionServiceOID.supply(),SERVICE_ID,
													 SERVICIO_MEDICO_TXT_ES,SERVICIO_MEDICO_TXT_EU);
				bloodDonationService = _clientApi.orgDivisionServicesAPI()
																  .getForCRUD()
																  .save(bloodDonationService);
			}
			outServices.add(bloodDonationService);
		}
		return outServices;
	}
	@Override
	protected Collection<AA14OrgDivisionServiceLocation> _loadOrCreateOrgDivisionServiceLocationsConfig(final AA14Organization org,
																  							      		final AA14OrgDivision division,
																  							      		final Collection<AA14OrgDivisionService> services) {
		Collection<AA14OrgDivisionServiceLocation> outLocs = Lists.newArrayListWithExpectedSize(3);
		
		// --- Location: Servicio mdico AR
		{
			AA14OrgDivisionServiceLocation donacionSangreAr = _clientApi.orgDivisionServiceLocationsAPI()
																			  .getForCRUD()
																			  .loadByIdOrNull(LOC_SERVICIO_MEDICO_AR_ID);
			if (donacionSangreAr == null) {
				log.warn("\t\t\t[Location]: SERVICIO MEDICO.Servicio mdico (AR) did NOT previously exists... creating it");
				donacionSangreAr = _buildLocation(org,
												  division,
												  this.getDonacionSangreService(),
												  AA14OrgDivisionServiceLocationOID.supply(),LOC_SERVICIO_MEDICO_AR_ID,
												  SERVICIO_MEDICO_TXT_ES + " (AR)",SERVICIO_MEDICO_TXT_EU + " (AR)",
												  _buildMedicalServiceArabaGeoPosition(),
												  _buildDonationsGasteizContactInfo(),
												  COLOR_SERVICIO_MEDICO);
				donacionSangreAr = _clientApi.orgDivisionServiceLocationsAPI()
															  .getForCRUD()
															  .save(donacionSangreAr);
			}
			outLocs.add(donacionSangreAr);
		}
		
		// --- Location: Servicio mdico GI
		{
			AA14OrgDivisionServiceLocation donacionSangreGi = _clientApi.orgDivisionServiceLocationsAPI()
																					  .getForCRUD()
																					  .loadByIdOrNull(LOC_SERVICIO_MEDICO_GI_ID);
			if (donacionSangreGi == null) {
				log.warn("\t\t\t[Location]: SERVICIO MEDICO.Servicio mdico (GI) did NOT previously exists... creating it");
				donacionSangreGi = _buildLocation(org,
												  division,
												  this.getDonacionSangreService(),
												  AA14OrgDivisionServiceLocationOID.supply(),LOC_SERVICIO_MEDICO_GI_ID,
												  SERVICIO_MEDICO_TXT_ES + " (GI)",SERVICIO_MEDICO_TXT_EU + " (GI)",
												  _buildMedicalServiceGipuzkoaGeoPosition(),
												  _buildDonationsGipuzkoaContactInfo(),
												  COLOR_SERVICIO_MEDICO);
				donacionSangreGi = _clientApi.orgDivisionServiceLocationsAPI()
															  .getForCRUD()
															  .save(donacionSangreGi);
			}
			outLocs.add(donacionSangreGi);
		}
		
		// --- Location: Servicio mdico BIZ
		{
			AA14OrgDivisionServiceLocation donacionSangreBiz = _clientApi.orgDivisionServiceLocationsAPI()
																					  .getForCRUD()
																					  .loadByIdOrNull(LOC_SERVICIO_MEDICO_BIZ_ID);
			if (donacionSangreBiz == null) {
				log.warn("\t\t\t[Location]: SERVICIO MEDICO.Servicio mdico (BIZ) did NOT previously exists... creating it");
				donacionSangreBiz = _buildLocation(org,
												   division,
												   this.getDonacionSangreService(),
												   AA14OrgDivisionServiceLocationOID.supply(),LOC_SERVICIO_MEDICO_BIZ_ID,
												   SERVICIO_MEDICO_TXT_ES + " (BIZ)",SERVICIO_MEDICO_TXT_EU + " (BIZ)",
												   _buildMedicalServiceBizkaiaGeoPosition(),
												   _buildMedicalServiceBizkaiaContactInfo(),
												   COLOR_SERVICIO_MEDICO);
				donacionSangreBiz = _clientApi.orgDivisionServiceLocationsAPI()
															  .getForCRUD()
															  .save(donacionSangreBiz);
			}
			outLocs.add(donacionSangreBiz);
		}
		
		return outLocs;
	}
	@Override
	protected Collection<AA14Schedule> _loadOrCreateSchedulesConfig(final AA14Organization org,
															  		final AA14OrgDivision division,
															  		final Collection<AA14OrgDivisionService> services,
															  		final Collection<AA14OrgDivisionServiceLocation> locs) {
		Collection<AA14Schedule> outSchs = Lists.newArrayListWithExpectedSize(3);
		
		// Schedule: Servicio mdico AR
		{
			AA14Schedule donacionSangreSchAr = _clientApi.schedulesAPI()
																	  .getForCRUD()
																	  .loadByIdOrNull(SCH_SERVICIO_MEDICO_AR_ID);
			if (donacionSangreSchAr == null) {
				log.warn("\t\t\t[Schedule]: SERVICIO MEDICO.Servicio mdico (AR) did NOT previously exists... creating it");
				donacionSangreSchAr = _buildSchedule(SCH_SERVICIO_MEDICO_AR_ID,
													 SERVICIO_MEDICO_TXT_ES + " SCHEDULE (AR)",SERVICIO_MEDICO_TXT_EU + " SCHEDULE (AR)",
													 _createDonationsScheduleBookingConfig(Time.of(12, 0),	 // day bookable range start 
															   							   Time.of(16, 30),// day bookable range end 
															   							   20,			 // slot length
															   							   1),			 // max appointments in slot
													 null,		// no qmatic orchestra config
													 this.getDonacionSangreArabaLocation());
				donacionSangreSchAr = _clientApi.schedulesAPI()
													  .getForCRUD()
													  .save(donacionSangreSchAr);
			}
			outSchs.add(donacionSangreSchAr);
		}
		// Schedule: Servicio mdico GI
		{
			AA14Schedule donacionSangreSchGi = _clientApi.schedulesAPI()
															  .getForCRUD()
															  .loadByIdOrNull(SCH_SERVICIO_MEDICO_GI_ID);
			if (donacionSangreSchGi == null) {
				log.warn("\t\t\t[Schedule]: SERVICIO MEDICO.Servicio mdico (AR) did NOT previously exists... creating it");
				donacionSangreSchGi = _buildSchedule(SCH_SERVICIO_MEDICO_GI_ID,
												     SERVICIO_MEDICO_TXT_ES + " SCHEDULE (GI)",SERVICIO_MEDICO_TXT_EU + " SCHEDULE (GI)",
												    _createDonationsScheduleBookingConfig(Time.of(12, 0),	 // day bookable range start 
															   							   Time.of(14, 20),// day bookable range end 
															   							   20,			 // slot length
															   							   1),			 // max appointments in slot
												     null,		// no qmatic orchestra config
												     this.getDonacionSangreGipuzkoaLocation());
				donacionSangreSchGi = _clientApi.schedulesAPI()
													  .getForCRUD()
													  .save(donacionSangreSchGi);
			}
			outSchs.add(donacionSangreSchGi);
		}
		// Schedule: Servicio medico BIZ
		{
			AA14Schedule medicalServiceSchBiz = _clientApi.schedulesAPI()
															  .getForCRUD()
															  .loadByIdOrNull(SCH_SERVICIO_MEDICO_BIZ_ID);
			if (medicalServiceSchBiz == null) {
				log.warn("\t\t\t[Schedule]: SERVICIO MEDICO.Servicio mdico (BIZ) did NOT previously exists... creating it");
				medicalServiceSchBiz = _buildSchedule(SCH_SERVICIO_MEDICO_BIZ_ID,
													  SERVICIO_MEDICO_TXT_ES + " SCHEDULE (BIZ)",SERVICIO_MEDICO_TXT_EU + " SCHEDULE (BIZ)",
													  _createDonationsScheduleBookingConfig(Time.of(8, 0),	 // day bookable range start 
															   							   Time.of(10, 00),// day bookable range end 
															   							   20,			 // slot length
															   							   1),			 // max appointments in slot
													 null,		// no qmatic orchestra config
													 this.getDonacionSangreBizkaiaLocation());
				medicalServiceSchBiz = _clientApi.schedulesAPI()
													  .getForCRUD()
													  .save(medicalServiceSchBiz);
			}
			outSchs.add(medicalServiceSchBiz);
		}
		
		
		return outSchs;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//	 
/////////////////////////////////////////////////////////////////////////////////////////
	private static AA14ScheduleBookingConfig _createDonationsScheduleBookingConfig(final Time dayBookableRangeStart,
																				   final Time dayBookableRangeEnd,
																				   final int slotLength,
																				   final int maxAppointmentsInSlot) {
		return new AA14ScheduleBookingConfig(dayBookableRangeStart,		// day bookable range start 
				    						 dayBookableRangeEnd,		// day bookable range end
											 slotLength,			// slot length
											 maxAppointmentsInSlot, // max appointments in slot
											 null); // future booking limit (date): 
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  Location addresses
/////////////////////////////////////////////////////////////////////////////////////////
	protected static GeoPosition _buildMedicalServiceArabaGeoPosition() {
		return GeoPosition.create()
						  .withCountry(SPAIN)
						  .withTerritory(EUSKADI_TERRITORY)
						  .withState(ARABA_STATE)
						  .withMunicipality(GASTEIZ)
						  .withStreet(GeoStreet.create()
									  	   .withNameInLang(Language.SPANISH,"Calle Donostia San Sebastin, 2 (Lakua 2) ") //
							  			   .withNameInLang(Language.BASQUE,"Donostia San Sebastian Kalea, 2 (Lakua 2) "))
						  .withZipCode(GeoZipCode.forId("01010"));
	}
	//FIXME put here the correct phoneNumbers
	protected static ContactInfo _buildDonationsGasteizContactInfo() {
		return ContactInfo.create()
						 		.addPhones(_buildZuzenenanPhones());
	}
	protected static GeoPosition _buildMedicalServiceGipuzkoaGeoPosition() {
		return GeoPosition.create()
						  .withCountry(SPAIN)
						  .withTerritory(EUSKADI_TERRITORY)
						  .withState(GIPUZKOA_STATE)
						  .withMunicipality(DONOSTIA)
						  .withStreet(GeoStreet.create()
									  	   .withNameInLang(Language.SPANISH,"Andia, 13 3 planta")
							  			   .withNameInLang(Language.BASQUE,"Andia, 13 3. solairua"))
						  .withZipCode(GeoZipCode.forId("20004"));
	}
	protected static ContactInfo _buildDonationsGipuzkoaContactInfo() {
		return ContactInfo.create()
						 		.addPhones(_buildZuzenenanPhones());
	}
	protected static GeoPosition _buildMedicalServiceBizkaiaGeoPosition() {
		return GeoPosition.create()
						  .withCountry(SPAIN)
						  .withTerritory(EUSKADI_TERRITORY)
						  .withState(BIZKAIA_STATE)
						  .withMunicipality(BILBAO)
						  .withStreet(GeoStreet.create()
									  	   .withNameInLang(Language.SPANISH,"Gran Va, 85 2 planta")
							  			   .withNameInLang(Language.BASQUE,"Gran Via kalea, 85 2. solairua"))
						  .withZipCode(GeoZipCode.forId("48011"));
	}
	protected static ContactInfo _buildMedicalServiceBizkaiaContactInfo() {
		return ContactInfo.create()
						 		.addPhones(_buildZuzenenanPhones());
	}
}
