package aa14f.client.api;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

import javax.inject.Inject;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Maps;

import aa14f.model.AA14OrgDivision;
import aa14f.model.AA14OrgDivisionService;
import aa14f.model.AA14OrgDivisionServiceLocation;
import aa14f.model.AA14OrgDivisionServiceLocationPresentationConfig;
import aa14f.model.AA14Organization;
import aa14f.model.AA14OrganizationalModelObjectRef;
import aa14f.model.AA14Schedule;
import aa14f.model.AA14ScheduleBookingConfig;
import aa14f.model.AA14ScheduleOrchestraConfig;
import aa14f.model.oids.AA14IDs.AA14OrgDivisionID;
import aa14f.model.oids.AA14IDs.AA14OrgDivisionServiceID;
import aa14f.model.oids.AA14IDs.AA14OrgDivisionServiceLocationID;
import aa14f.model.oids.AA14IDs.AA14OrganizationID;
import aa14f.model.oids.AA14IDs.AA14ScheduleID;
import aa14f.model.oids.AA14OIDs.AA14OrgDivisionOID;
import aa14f.model.oids.AA14OIDs.AA14OrgDivisionServiceLocationOID;
import aa14f.model.oids.AA14OIDs.AA14OrgDivisionServiceOID;
import aa14f.model.oids.AA14OIDs.AA14OrganizationOID;
import aa14f.model.oids.AA14OIDs.AA14ScheduleOID;
import aa14f.model.summaries.AA14SummarizedOrgHierarchy;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import r01f.locale.Language;
import r01f.locale.LanguageTexts.LangTextNotFoundBehabior;
import r01f.locale.services.LanguageTextsBuilder;
import r01f.patterns.Memoized;
import r01f.types.Color;
import r01f.types.contact.ContactInfo;
import r01f.types.geo.GeoCountry;
import r01f.types.geo.GeoMunicipality;
import r01f.types.geo.GeoPosition;
import r01f.types.geo.GeoState;
import r01f.types.geo.GeoTerritory;

@Accessors(prefix="_")
@Slf4j
abstract class AA14ConfigBase {
/////////////////////////////////////////////////////////////////////////////////////////
//  CONSTANTS
/////////////////////////////////////////////////////////////////////////////////////////
	public static final AA14OrganizationID ORG_ID = AA14OrganizationID.forId("EJGV");
	
	public static final GeoCountry SPAIN = GeoCountry.create()
											  		 .withNameInLang(Language.SPANISH,"Espaa")
											  		 .withNameInLang(Language.BASQUE,"Espania");
	public static final GeoTerritory EUSKADI_TERRITORY = GeoTerritory.create()
														  		 .withNameInLang(Language.SPANISH,"Euskadi")
														  		 .withNameInLang(Language.BASQUE,"Euskadi");
//	public static final GeoTerritory BIZKAIA_TERRITORY = GeoTerritory.create()
//															  		 .withNameInLang(Language.SPANISH,"Bizkaia")
//															  		 .withNameInLang(Language.BASQUE,"Bizkaia");
	public static final GeoState BIZKAIA_STATE = GeoState.create()
												  		  .withNameInLang(Language.SPANISH,"Bizkaia")
												  		  .withNameInLang(Language.BASQUE,"Bizkaia");
	public static final GeoMunicipality BILBAO = GeoMunicipality.create()
												  			 	.withNameInLang(Language.SPANISH,"Bilbao")
												  			 	.withNameInLang(Language.BASQUE,"Bilbao");
//	public static final GeoTerritory ARABA_TERRITORY = GeoTerritory.create()
//															  		 .withNameInLang(Language.SPANISH,"Araba/lava")
//															  		 .withNameInLang(Language.BASQUE,"Araba/lava");
	public static final GeoState ARABA_STATE = GeoState.create()
												  		  .withNameInLang(Language.SPANISH,"Araba/lava")
												  		  .withNameInLang(Language.BASQUE,"Araba/lava");
	public static final GeoMunicipality GASTEIZ = GeoMunicipality.create()
													  			 	.withNameInLang(Language.SPANISH,"Vitoria-Gasteiz")
													  			 	.withNameInLang(Language.BASQUE,"Vitoria-Gasteiz");
//	public static final GeoTerritory GIPUZKOA_TERRITORY = GeoTerritory.create()
//																  		 .withNameInLang(Language.SPANISH,"Gipuzkoa")
//																  		 .withNameInLang(Language.BASQUE,"Gipuzkoa");
	public static final GeoState GIPUZKOA_STATE = GeoState.create()
													  		  .withNameInLang(Language.SPANISH,"Gipuzkoa")
													  		  .withNameInLang(Language.BASQUE,"Gipuzkoa");
	public static final GeoMunicipality DONOSTIA = GeoMunicipality.create()
													  			 	.withNameInLang(Language.SPANISH,"Donostia-San Sebastin")
													  			 	.withNameInLang(Language.BASQUE,"Donostia-San Sebastin");
	
/////////////////////////////////////////////////////////////////////////////////////////
//  INJECTED FIELDS
/////////////////////////////////////////////////////////////////////////////////////////
	@Inject
	protected AA14ClientAPI _clientApi;

	
/////////////////////////////////////////////////////////////////////////////////////////
//  OIDs initialized at ensureConfig() method
/////////////////////////////////////////////////////////////////////////////////////////
//	private AA14OrganizationOID _orgOid;
//	private AA14OrgDivisionOID _orgDivisionOid;
//	private Collection<AA14OrgDivisionServiceOID> _orgDivisionServicesOids;
//	private Collection<AA14OrgDivisionServiceLocationOID> _orgDivisionServiceLocationsOids;
//	private Collection<AA14ScheduleOID> _schedulesOids;
/////////////////////////////////////////////////////////////////////////////////////////
//  MEMOIZED FIELDS TO AVOID DB ACCESS
/////////////////////////////////////////////////////////////////////////////////////////
	@Getter(AccessLevel.PROTECTED) 
	protected final Memoized<AA14Organization> _memoizedOrganization = new Memoized<AA14Organization>() {
																				@Override
																				protected AA14Organization supply() {
																					return _loadOrCreateOrganizationConfig();
																				}
																	   };
	@Getter(AccessLevel.PROTECTED) 
	private final Memoized<AA14OrgDivision> _memoizedDivision = new Memoized<AA14OrgDivision>() {
																			@Override
																			protected AA14OrgDivision supply() {
																				AA14Organization org = AA14ConfigBase.this.getMemoizedOrganization().get();
																				return _loadOrCreateOrgDivisionConfig(org);
																			}
																 };
	@Getter(AccessLevel.PROTECTED) 
	private final Memoized<Map<AA14OrgDivisionServiceOID,AA14OrgDivisionService>> _memoizedServicesByOid = 
						new Memoized<Map<AA14OrgDivisionServiceOID,AA14OrgDivisionService>>() {
								@Override
								protected Map<AA14OrgDivisionServiceOID,AA14OrgDivisionService> supply() {
									AA14Organization org = AA14ConfigBase.this.getMemoizedOrganization().get();
									AA14OrgDivision div = AA14ConfigBase.this.getMemoizedDivision().get();
									Collection<AA14OrgDivisionService> services = _loadOrCreateOrgDivisionServicesConfig(org,div);
									Map<AA14OrgDivisionServiceOID,AA14OrgDivisionService> outMap = Maps.newHashMapWithExpectedSize(services.size());
									for (AA14OrgDivisionService srvc : services) {
										outMap.put(srvc.getOid(),srvc);
									}
									return outMap;
								}
						};
	@Getter(AccessLevel.PROTECTED) 
	private final Memoized<Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation>> _memoizedLocationsByOid = 
						new Memoized<Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation>>() {
								@Override
								protected Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation> supply() {
									AA14Organization org = AA14ConfigBase.this.getMemoizedOrganization().get();
									AA14OrgDivision div = AA14ConfigBase.this.getMemoizedDivision().get();
									Collection<AA14OrgDivisionService> services = AA14ConfigBase.this.getMemoizedServicesByOid().get().values();
									Collection<AA14OrgDivisionServiceLocation> locs = _loadOrCreateOrgDivisionServiceLocationsConfig(org,div,services);
									Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation> outMap = Maps.newHashMapWithExpectedSize(locs.size());
									for (AA14OrgDivisionServiceLocation loc : locs) {
										outMap.put(loc.getOid(),loc);
									}
									return outMap;
								}
						};
	@Getter(AccessLevel.PROTECTED) 
	private final Memoized<Map<AA14ScheduleOID,AA14Schedule>> _memoizedSchedulesByOid = 
						new Memoized<Map<AA14ScheduleOID,AA14Schedule>>() {
								@Override
								protected Map<AA14ScheduleOID,AA14Schedule> supply() {
									AA14Organization org = AA14ConfigBase.this.getMemoizedOrganization().get();
									AA14OrgDivision div = AA14ConfigBase.this.getMemoizedDivision().get();
									Collection<AA14OrgDivisionService> services = AA14ConfigBase.this.getMemoizedServicesByOid().get().values();
									Collection<AA14OrgDivisionServiceLocation> locs = AA14ConfigBase.this.getMemoizedLocationsByOid().get().values();
									Collection<AA14Schedule> schs = _loadOrCreateSchedulesConfig(org,div,services,locs);
									Map<AA14ScheduleOID,AA14Schedule> outMap = Maps.newHashMapWithExpectedSize(schs.size());
									for (AA14Schedule sch : schs) {
										outMap.put(sch.getOid(),sch);
									}
									return outMap;
								}
						};
/////////////////////////////////////////////////////////////////////////////////////////
//  GETTERS
/////////////////////////////////////////////////////////////////////////////////////////						
	public AA14Organization getOrganization() {
		return this.getMemoizedOrganization().get();
	}
	public AA14OrgDivision getDivision() {
		return this.getMemoizedDivision().get();
	}
	public Collection<AA14OrgDivisionService> getServices() {
		return this.getMemoizedServicesByOid().get()
											  .values();
	}
	public AA14OrgDivisionService getServiceFor(final AA14OrgDivisionServiceOID srvcOid) {
		return this.getMemoizedServicesByOid().get()
											  .get(srvcOid);
	}
	public AA14OrgDivisionService getServiceFor(final AA14OrgDivisionServiceID srvcId) {
		return FluentIterable.from(this.getMemoizedServicesByOid().get().values())
							 .filter(new Predicate<AA14OrgDivisionService>() {
												@Override
												public boolean apply(final AA14OrgDivisionService srvc) {
													return srvc.getId().equals(srvcId);
												}
							 		 })
							 .first().orNull();
	}
	public Collection<AA14OrgDivisionServiceLocation> getLocations() {
		return this.getMemoizedLocationsByOid().get()
											   .values();
	}
	public AA14OrgDivisionServiceLocation getLocationFor(final AA14OrgDivisionServiceLocationOID locOid) {
		return this.getMemoizedLocationsByOid().get()
							  		   		   .get(locOid);
	}
	public AA14OrgDivisionServiceLocation getLocationFor(final AA14OrgDivisionServiceLocationID locId) {
		return FluentIterable.from(this.getMemoizedLocationsByOid().get().values())
							 .filter(new Predicate<AA14OrgDivisionServiceLocation>() {
												@Override
												public boolean apply(final AA14OrgDivisionServiceLocation loc) {
													return loc.getId().equals(locId);
												}
							 		 })
							 .first().orNull();
	}
	public Collection<AA14Schedule> getSchedules() {
		return this.getMemoizedSchedulesByOid().get()
											   .values();
	}
	public AA14Schedule getScheduleFor(final AA14ScheduleOID schOid) {
		return this.getMemoizedSchedulesByOid().get()
							  		   		   .get(schOid);
	}
	public AA14Schedule getScheduleFor(final AA14ScheduleID schId) {
		return FluentIterable.from(this.getMemoizedSchedulesByOid().get().values())
							 .filter(new Predicate<AA14Schedule>() {
												@Override
												public boolean apply(final AA14Schedule sch) {
													return sch.getId().equals(schId);
												}
							 		 })
							 .first().orNull();
	}
	public Map<AA14ScheduleOID,AA14Schedule> getSchedulesByOid() {
		return this.getMemoizedSchedulesByOid().get();
	}
	public Map<AA14ScheduleID,AA14Schedule> getSchedulesById() {
		Map<AA14ScheduleOID,AA14Schedule> schsByOid = this.getSchedulesByOid();
		Map<AA14ScheduleID,AA14Schedule> outMap = Maps.newHashMapWithExpectedSize(schsByOid.size());
		for (AA14Schedule sch : schsByOid.values()) {
			outMap.put(sch.getId(),sch);
		}
		return outMap;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  INIT
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Ensures the org exists
	 * @return
	 */
	protected abstract AA14Organization _loadOrCreateOrganizationConfig();
	/**
	 * Ensures the org division exists
	 * @param org
	 * @return
	 */
	protected abstract AA14OrgDivision _loadOrCreateOrgDivisionConfig(final AA14Organization org);
	/**
	 * Ensures the services config exists
	 * @param org
	 * @param division
	 * @return
	 */
	protected abstract Collection<AA14OrgDivisionService> _loadOrCreateOrgDivisionServicesConfig(final AA14Organization org,
																								 final AA14OrgDivision division);
	/**
	 * Ensures the locations config exists
	 * @param org
	 * @param division
	 * @param services
	 * @return
	 */
	protected abstract Collection<AA14OrgDivisionServiceLocation> _loadOrCreateOrgDivisionServiceLocationsConfig(final AA14Organization org,
																  							            	     final AA14OrgDivision division,
																  							            	     final Collection<AA14OrgDivisionService> services);
	/**
	 * Ensures the schedules config exists
	 * @param org
	 * @param division
	 * @param services
	 * @param locs
	 * @return
	 */
	protected abstract Collection<AA14Schedule> _loadOrCreateSchedulesConfig(final AA14Organization org,
															  				 final AA14OrgDivision division,
															  				 final Collection<AA14OrgDivisionService> services,
															  				 final Collection<AA14OrgDivisionServiceLocation> locs);
	/**
	 * Ensures that the organization, division, service and location
	 * exists for the trafikoa service
	 * (when in web environment it's called from the servlet context listener; when in test environment it's called from the test bootstrap)
	 */
	public void ensureConfig() {
		// force the creation of the organizational structure
		this.getMemoizedOrganization().get();
		this.getMemoizedDivision().get();
		this.getMemoizedLocationsByOid().get();
		this.getMemoizedSchedulesByOid().get();
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  
/////////////////////////////////////////////////////////////////////////////////////////
	protected static AA14Organization _buildOrganization(final AA14OrganizationOID oid,final AA14OrganizationID id,
													     final String nameES,final String nameEU) {
		AA14Organization outOrg = new AA14Organization();
		outOrg.setOid(oid);
		outOrg.setId(id);
		outOrg.setNameByLanguage(LanguageTextsBuilder.createMapBacked()
										   .withMissingLangTextBehavior(LangTextNotFoundBehabior.RETURN_NULL)
										   .addForLang(Language.SPANISH,nameES)
										   .addForLang(Language.BASQUE,nameEU)
										   .build());		
		return outOrg;
	}
	protected static AA14OrgDivision _buildDivision(final AA14Organization org,
												  	final AA14OrgDivisionOID oid,final AA14OrgDivisionID id,
												  	final String nameES,final String nameEU) {
		AA14OrgDivision outDiv = new AA14OrgDivision();
		outDiv.setOid(oid);
		outDiv.setId(id);
		outDiv.setOrgRef(new AA14OrganizationalModelObjectRef<AA14OrganizationOID,AA14OrganizationID>(org.getOid(),org.getId()));
		outDiv.setNameByLanguage(LanguageTextsBuilder.createMapBacked()
										   .withMissingLangTextBehavior(LangTextNotFoundBehabior.RETURN_NULL)
										   .addForLang(Language.SPANISH,nameES)
										   .addForLang(Language.BASQUE,nameEU)
										   .build());
		return outDiv;
	}
	protected static AA14OrgDivisionService _buildService(final AA14Organization org,
												 		  final AA14OrgDivision division,
												 		  final AA14OrgDivisionServiceOID oid,final AA14OrgDivisionServiceID id,
												 		  final String nameES,final String nameEU) {
		AA14OrgDivisionService outService = new AA14OrgDivisionService();
		outService.setOid(oid);
		outService.setId(id);
		outService.setOrgRef(new AA14OrganizationalModelObjectRef<AA14OrganizationOID,AA14OrganizationID>(org.getOid(),org.getId()));
		outService.setOrgDivisionRef(new AA14OrganizationalModelObjectRef<AA14OrgDivisionOID,AA14OrgDivisionID>(division.getOid(),division.getId()));
		outService.setNameByLanguage(LanguageTextsBuilder.createMapBacked()
											     .withMissingLangTextBehavior(LangTextNotFoundBehabior.RETURN_NULL)
											     .addForLang(Language.SPANISH,nameES)
											     .addForLang(Language.BASQUE,nameEU)
											     .build());
		return outService;
	}
	protected static AA14OrgDivisionServiceLocation _buildLocation(final AA14Organization org,
												 		  		   final AA14OrgDivision division,
												 		  		   final AA14OrgDivisionService service,
												 		  		   final AA14OrgDivisionServiceLocationOID oid,final AA14OrgDivisionServiceLocationID locId,
												 		  		   final String locNameES,final String locNameEU,
												 		  		   final GeoPosition geoPosition,
												 		  		   final ContactInfo contactInfo,
												 		  		   final Color presentationColor) {
		AA14OrgDivisionServiceLocation outLoc = new AA14OrgDivisionServiceLocation();
		outLoc.setOid(oid);
		outLoc.setId(locId);
		outLoc.setOrgRef(new AA14OrganizationalModelObjectRef<AA14OrganizationOID,AA14OrganizationID>(org.getOid(),org.getId()));
		outLoc.setOrgDivisionRef(new AA14OrganizationalModelObjectRef<AA14OrgDivisionOID,AA14OrgDivisionID>(division.getOid(),division.getId()));
		outLoc.setOrgDivisionServiceRef(new AA14OrganizationalModelObjectRef<AA14OrgDivisionServiceOID,AA14OrgDivisionServiceID>(service.getOid(),service.getId()));
		outLoc.setNameByLanguage(LanguageTextsBuilder.createMapBacked()
										     .withMissingLangTextBehavior(LangTextNotFoundBehabior.RETURN_NULL)
										     .addForLang(Language.SPANISH,locNameES)
										     .addForLang(Language.BASQUE,locNameEU)
										     .build());
		outLoc.setPosition(geoPosition);
		outLoc.setContactInfo(contactInfo);
		outLoc.setPresentationConfig(new AA14OrgDivisionServiceLocationPresentationConfig(presentationColor));
		return outLoc;
	}
	protected static AA14Schedule _buildSchedule(final AA14ScheduleID scheduleId,
												 final String schES,final String schEU,
												 final AA14ScheduleBookingConfig schConfig,final AA14ScheduleOrchestraConfig orchestraConfig,
												 final AA14OrgDivisionServiceLocation... serviceLocs) {
		return _buildSchedule(scheduleId, 
							  schES,schEU,
							  schConfig,orchestraConfig,
							  Arrays.asList(serviceLocs));
	}
	protected static AA14Schedule _buildSchedule(final AA14ScheduleID scheduleId,
												 final String schES,final String schEU,
												 final AA14ScheduleBookingConfig schConfig,final AA14ScheduleOrchestraConfig orchestraConfig,
												 final Collection<AA14OrgDivisionServiceLocation> serviceLocs) {
		AA14ScheduleOID oid = AA14ScheduleOID.supply();
		AA14Schedule outSch = new AA14Schedule();
		outSch.setOid(oid);
		outSch.setId(scheduleId);
		outSch.setNameByLanguage(LanguageTextsBuilder.createMapBacked()
										     .withMissingLangTextBehavior(LangTextNotFoundBehabior.RETURN_NULL)
										     .addForLang(Language.SPANISH,schES)
										     .addForLang(Language.BASQUE,schEU)
										     .build());
		outSch.setServiceLocationsRefs(FluentIterable.from(serviceLocs)
													 .transform(new Function<AA14OrgDivisionServiceLocation,
															 				 AA14OrganizationalModelObjectRef<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocationID>>() {
																		@Override
																		public AA14OrganizationalModelObjectRef<AA14OrgDivisionServiceLocationOID, AA14OrgDivisionServiceLocationID> apply(final AA14OrgDivisionServiceLocation loc) {
																			return loc.getReference();
																		}
													 			})
													 .toList());
		outSch.setBookingConfig(schConfig);
		outSch.setOrchestraConfig(orchestraConfig);
		
		return outSch;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  HIERARCHY
/////////////////////////////////////////////////////////////////////////////////////////	
	public AA14SummarizedOrgHierarchy getSummarizedOrgHierarchyFor(final AA14OrgDivisionServiceLocationOID locOid,
																   final Language lang) {
		Map<Language,AA14SummarizedOrgHierarchy> orgHierarchy = _summarizedOrgHierarchyByLocationOid.get()
							  		   				  				.get(locOid);
		return orgHierarchy != null ? orgHierarchy.get(lang) : null;
	}
	public AA14SummarizedOrgHierarchy getSummarizedOrgHierarchyFor(final AA14OrgDivisionServiceLocationID locId,
																   final Language lang) {
		return AA14ConfigBase.filterOrgHierarchyForLocationWithId(_summarizedOrgHierarchyByLocationOid.get(),
																  locId,
																  lang);
	}
	@Getter private Memoized<Map<AA14OrgDivisionServiceLocationOID,
					 	 	     Map<Language,AA14SummarizedOrgHierarchy>>> _summarizedOrgHierarchyByLocationOid = 
					 	 new Memoized<Map<AA14OrgDivisionServiceLocationOID,
					 					  Map<Language,AA14SummarizedOrgHierarchy>>>() {
									@Override
									protected Map<AA14OrgDivisionServiceLocationOID,
												  Map<Language,AA14SummarizedOrgHierarchy>> supply() {
										Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation> locsByOid = _memoizedLocationsByOid.get();
										Map<AA14OrgDivisionServiceLocationOID,
											Map<Language,AA14SummarizedOrgHierarchy>> outMap = Maps.newHashMapWithExpectedSize(locsByOid.size());
										
										for (AA14OrgDivisionServiceLocation loc : locsByOid.values()) {
											outMap.put(loc.getOid(),
													   _orgHierarchy(loc.getOrgRef().getOid(),
																	 loc.getOrgDivisionRef().getOid(),
																	 loc.getOrgDivisionServiceRef().getOid(),
																	 loc.getOid()));
										}
										return outMap;
									}
						};
	public static AA14SummarizedOrgHierarchy filterOrgHierarchyForLocationWithId(final Map<AA14OrgDivisionServiceLocationOID,
					 	 	     														   Map<Language,AA14SummarizedOrgHierarchy>> summarizedOrgHierarchyByLocationOid,
					 	 	     												 final AA14OrgDivisionServiceLocationID locId,
					 	 	     												 final Language lang) {
		Map<Language,AA14SummarizedOrgHierarchy> orgHierarchy = FluentIterable.from(summarizedOrgHierarchyByLocationOid.values())
																			  .filter(new Predicate<Map<Language,AA14SummarizedOrgHierarchy>>() {
																							@Override
																							public boolean apply(final Map<Language,AA14SummarizedOrgHierarchy> sumByLang) {
																								return sumByLang.get(lang) != null
																									&& sumByLang.get(lang).getLocation().getId().is(locId);
																							}
																			 		  })
																			  .first().orNull();
		if (orgHierarchy == null) log.error("There does NOT exists a location with id={}",locId);
		return orgHierarchy != null ? orgHierarchy.get(lang) : null;
	}
	protected Map<Language,AA14SummarizedOrgHierarchy> _orgHierarchy(final AA14OrganizationOID orgOid,final AA14OrgDivisionOID divOid,final AA14OrgDivisionServiceOID srvcOid,final AA14OrgDivisionServiceLocationOID locOid) {
		Map<Language,AA14SummarizedOrgHierarchy> outMap = Maps.newHashMapWithExpectedSize(2);
		outMap.put(Language.SPANISH,
				   _orgHierarchy(orgOid,divOid,srvcOid,locOid,
						   		 Language.SPANISH));
		outMap.put(Language.BASQUE,
				   _orgHierarchy(orgOid,divOid,srvcOid,locOid,
						   		 Language.BASQUE));
		return outMap;
	}
	
	protected AA14SummarizedOrgHierarchy _orgHierarchy(final AA14OrganizationOID orgOid,final AA14OrgDivisionOID divOid,final AA14OrgDivisionServiceOID srvcOid,final AA14OrgDivisionServiceLocationOID locOid,
													   final Language lang) {
		AA14SummarizedOrgHierarchy outHierarchy = new AA14SummarizedOrgHierarchy();
		
		AA14Organization org = this.getOrganization();
		AA14OrgDivision div = this.getDivision();
		AA14OrgDivisionService srvc = this.getServiceFor(srvcOid);
		AA14OrgDivisionServiceLocation loc = this.getLocationFor(locOid);
		
		outHierarchy.setOrganization(org.getSummarizedIn(lang));
		outHierarchy.setDivision(div.getSummarizedIn(lang));
		outHierarchy.setService(srvc.getSummarizedIn(lang));
		outHierarchy.setLocation(loc.getSummarizedIn(lang));
		return outHierarchy;
	}
}
