package aa14f.client.api;

import java.util.Collection;
import java.util.Map;

import javax.inject.Named;
import javax.inject.Singleton;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.inject.Inject;
import com.google.inject.Provider;

import aa14f.api.interfaces.AA14NotifierServices;
import aa14f.api.interfaces.AA14SearchServices;
import aa14f.client.api.sub.AA14ClientAPIForBookedSlots;
import aa14f.client.api.sub.AA14ClientAPIForNotifier;
import aa14f.client.api.sub.AA14ClientAPIForOrgDivisionServiceLocations;
import aa14f.client.api.sub.AA14ClientAPIForOrgDivisionServices;
import aa14f.client.api.sub.AA14ClientAPIForOrgDivisions;
import aa14f.client.api.sub.AA14ClientAPIForOrganizations;
import aa14f.client.api.sub.AA14ClientAPIForSchedules;
import aa14f.client.api.sub.AA14ClientAPIForSearch;
import aa14f.common.internal.AA14AppCodes;
import aa14f.model.AA14OrgDivisionServiceLocation;
import aa14f.model.AA14Schedule;
import aa14f.model.AA14ScheduleBookingConfig;
import aa14f.model.oids.AA14IDs.AA14OrgDivisionServiceLocationID;
import aa14f.model.oids.AA14IDs.AA14ScheduleID;
import aa14f.model.oids.AA14OIDs.AA14OrgDivisionServiceLocationOID;
import aa14f.model.oids.AA14OIDs.AA14ScheduleOID;
import aa14f.model.summaries.AA14SummarizedOrgHierarchy;
import lombok.Getter;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import r01f.locale.Language;
import r01f.model.annotations.ModelObjectsMarshaller;
import r01f.objectstreamer.Marshaller;
import r01f.patterns.Memoized;
import r01f.securitycontext.SecurityContext;
import r01f.services.client.ClientAPIImplBase;
import r01f.services.interfaces.ServiceInterface;
import r01f.util.types.collections.CollectionUtils;



/**
 * Base type for every API implementation of appointments service.
 */
@Singleton
@Slf4j
@Accessors(prefix="_")
public class AA14ClientAPI
     extends ClientAPIImplBase {
/////////////////////////////////////////////////////////////////////////////////////////
//  INJECTED FIELDS
/////////////////////////////////////////////////////////////////////////////////////////
	@Inject
	@Getter private AA14ConfigForTrafikoa _configForTrafikoa;
	
	@Inject
	@Getter private AA14ConfigForBizilagun _configForBizilagun;
	
	@Inject
	@Getter private AA14ConfigForBloodDonation _configForBloodDonation;
	
	@Inject
	@Getter private AA14ConfigForMedicalService _configForMedicalService;
/////////////////////////////////////////////////////////////////////////////////////////
//  ORG SUB-APIs (created at the constructor)
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Organizations
	 */
	private AA14ClientAPIForOrganizations _organizationsAPI;
	/**
	 * Divisions
	 */
	private AA14ClientAPIForOrgDivisions _orgDivisionsAPI;
	/**
	 * Services
	 */
	private AA14ClientAPIForOrgDivisionServices _orgDivisionServicesAPI;
	/**
	 * Locations
	 */
	private AA14ClientAPIForOrgDivisionServiceLocations _orgDivisionServiceLocationsAPI;
	/**
	 * Schedules
	 */
	private AA14ClientAPIForSchedules _schedulesAPI;
/////////////////////////////////////////////////////////////////////////////////////////
//  APPOINTMENTS SUB-APIs
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Appointments
	 */
	private AA14ClientAPIForBookedSlots _bookedSlotsAPI;
/////////////////////////////////////////////////////////////////////////////////////////
//  SEARCH SUB-APIS
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Entity search API
	 */
	private AA14ClientAPIForSearch _searchAPI;
/////////////////////////////////////////////////////////////////////////////////////////
//	NOTIFICATION SUB-API 
/////////////////////////////////////////////////////////////////////////////////////////	
	private AA14ClientAPIForNotifier _notifierAPI;
	
/////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTOR
/////////////////////////////////////////////////////////////////////////////////////////
	@Inject @SuppressWarnings("rawtypes")
	public AA14ClientAPI(						 final Provider<SecurityContext> securityContextProvider,
						 @ModelObjectsMarshaller final Marshaller modelObjectsMarshaller,
					     @Named(AA14AppCodes.API_APPCODE_STR) final Map<Class,ServiceInterface> srvcIfaceMappings) {	// comes from injection
		// Services proxy
		super(securityContextProvider,
			  modelObjectsMarshaller,
			  srvcIfaceMappings);

		// Build every sub-api
		_organizationsAPI = new AA14ClientAPIForOrganizations(securityContextProvider,
															  modelObjectsMarshaller,
															  srvcIfaceMappings);						    
		_orgDivisionsAPI = new AA14ClientAPIForOrgDivisions(securityContextProvider,
															modelObjectsMarshaller,
				  									        srvcIfaceMappings);
		_orgDivisionServicesAPI = new AA14ClientAPIForOrgDivisionServices(securityContextProvider,
																		  modelObjectsMarshaller,
				  														  srvcIfaceMappings);
		_orgDivisionServiceLocationsAPI = new AA14ClientAPIForOrgDivisionServiceLocations(securityContextProvider,
																						  modelObjectsMarshaller,
																						  srvcIfaceMappings);
		_schedulesAPI = new AA14ClientAPIForSchedules(securityContextProvider,
													  modelObjectsMarshaller,
													  srvcIfaceMappings);
		
		_bookedSlotsAPI = new AA14ClientAPIForBookedSlots(securityContextProvider,
														  modelObjectsMarshaller,
														  srvcIfaceMappings);
		
		_searchAPI = new AA14ClientAPIForSearch(securityContextProvider,
												modelObjectsMarshaller,
												this.getServiceInterfaceCoreImplOrProxy(AA14SearchServices.class));
		
		_notifierAPI = new AA14ClientAPIForNotifier(securityContextProvider,
													modelObjectsMarshaller,
													this.getServiceInterfaceCoreImplOrProxy(AA14NotifierServices.class));
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  SUB-APIs
/////////////////////////////////////////////////////////////////////////////////////////
	public AA14ClientAPIForOrganizations organizationsAPI() {
		return _organizationsAPI;
	}
	public AA14ClientAPIForOrgDivisions orgDivisionsAPI() {
		return _orgDivisionsAPI;
	}
	public AA14ClientAPIForOrgDivisionServices orgDivisionServicesAPI() {
		return _orgDivisionServicesAPI;
	}
	public AA14ClientAPIForOrgDivisionServiceLocations orgDivisionServiceLocationsAPI() {
		return _orgDivisionServiceLocationsAPI;
	}
	public AA14ClientAPIForSchedules schedulesAPI() {
		return _schedulesAPI;
	}
	public AA14ClientAPIForBookedSlots bookedSlotsAPI() {
		return _bookedSlotsAPI;
	}	
	public AA14ClientAPIForSearch searchAPI() {
		return _searchAPI;
	}
	public AA14ClientAPIForNotifier notifierAPI() {
		return _notifierAPI;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  
/////////////////////////////////////////////////////////////////////////////////////////
	private Memoized<Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation>> _locationsByOid = 
				new Memoized<Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation>>() {
						@Override
						protected Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation> supply() {
							Iterable<AA14OrgDivisionServiceLocation> allLocs = Iterables.concat(_configForTrafikoa.getMemoizedLocationsByOid().get().values(),
																								_configForBizilagun.getMemoizedLocationsByOid().get().values(),
																								_configForBloodDonation.getMemoizedLocationsByOid().get().values(),
																								_configForMedicalService.getMemoizedLocationsByOid().get().values());
							Map<AA14OrgDivisionServiceLocationOID,AA14OrgDivisionServiceLocation> outMap = Maps.newLinkedHashMap();
							for (AA14OrgDivisionServiceLocation loc : allLocs) outMap.put(loc.getOid(),loc);
							return outMap;
						}
				};
	/**
	 * Loads the location hierarchy by it's oid
	 * @param locOid
	 * @param lang
	 * @return
	 */
	public AA14OrgDivisionServiceLocation getLocationFor(final AA14OrgDivisionServiceLocationOID locOid) {
		AA14OrgDivisionServiceLocation outLoc = _locationsByOid.get()
							  								   .get(locOid);
		if (outLoc == null) log.error("There does NOT exists a location with oid={}",locOid);
		return outLoc;
	}
	/**
	 * Loads the location hierarchy by it's id
	 * @param locId
	 * @param lang
	 * @return
	 */
	public AA14OrgDivisionServiceLocation getLocationFor(final AA14OrgDivisionServiceLocationID locId) {
		AA14OrgDivisionServiceLocation outLoc = FluentIterable.from(_locationsByOid.get().values())
													  .filter(new Predicate<AA14OrgDivisionServiceLocation>() {
																	@Override
																	public boolean apply(final AA14OrgDivisionServiceLocation loc) {
																		return loc.getId().is(locId);
																	}
													  		  })
													  .first().orNull();
		if (outLoc == null) log.error("There does NOT exists a location with id={}",locId);
		return outLoc;
	}
	/**
	 * Returns the locations for a given schedule
	 * @param schId
	 * @return
	 */
	public Collection<AA14OrgDivisionServiceLocationOID> getLocationsOidsFor(final AA14ScheduleID schId) {
		AA14Schedule sch = this.getScheduleFor(schId);
		return sch.getServiceLocationsOids();
	}
	/**
	 * Returns the locations for a given schedule
	 * @param schId
	 * @return
	 */
	public Collection<AA14OrgDivisionServiceLocationID> getLocationsIdsFor(final AA14ScheduleID schId) {
		AA14Schedule sch = this.getScheduleFor(schId);
		return sch.getServiceLocationsIds();
	}
/////////////////////////////////////////////////////////////////////////////////////////
//	 
/////////////////////////////////////////////////////////////////////////////////////////
	private Memoized<Map<AA14ScheduleOID,AA14Schedule>> _schedulesByOid = 
				new Memoized<Map<AA14ScheduleOID,AA14Schedule>>() {
						@Override
						protected Map<AA14ScheduleOID,AA14Schedule> supply() {
							Iterable<AA14Schedule> allSchs = Iterables.concat(_configForTrafikoa.getMemoizedSchedulesByOid().get().values(),
																			  _configForBizilagun.getMemoizedSchedulesByOid().get().values(),
																			  _configForBloodDonation.getMemoizedSchedulesByOid().get().values(),
																			  _configForMedicalService.getMemoizedSchedulesByOid().get().values());
							Map<AA14ScheduleOID,AA14Schedule> outMap = Maps.newLinkedHashMap();
							for (AA14Schedule sch : allSchs) outMap.put(sch.getOid(),sch);
							return outMap;
						}
				};
	/**
	 * Loads the location hierarchy by it's oid
	 * @param schOid
	 * @param lang
	 * @return
	 */
	public AA14Schedule getScheduleFor(final AA14ScheduleOID schOid) {
		AA14Schedule outSch = _schedulesByOid.get()
											 .get(schOid);
		if (outSch == null) log.error("There does NOT exists a schedule with oid={}",schOid);
		return outSch;
	}
	/**
	 * Loads the location hierarchy by it's id
	 * @param schId
	 * @param lang
	 * @return
	 */
	public AA14Schedule getScheduleFor(final AA14ScheduleID schId) {
		AA14Schedule outSch = FluentIterable.from(_schedulesByOid.get().values())
								  .filter(new Predicate<AA14Schedule>() {
												@Override
												public boolean apply(final AA14Schedule sch) {
													return sch.getId().is(schId);
												}
								  		  })
								  .first().orNull();
		if (outSch == null) log.error("There does NOT exists a schedule with id={}",schId);
		return outSch;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//	 
/////////////////////////////////////////////////////////////////////////////////////////
	private AA14ScheduleBookingConfig _locationScheduleBookingConfigFor(final AA14OrgDivisionServiceLocation loc) {
		Collection<AA14ScheduleOID> schOids = loc.getSchedulesOids();
		if (CollectionUtils.isNullOrEmpty(schOids)) throw new IllegalStateException("The location with oid/id=" + loc.getOid() + "/" + loc.getId() + " does NOT have any schedule!!");
								
		// load all schedules and ensure they all have the same booking config
		Collection<AA14ScheduleBookingConfig> bookingCfgs = FluentIterable.from(schOids)
																	.transform(new Function<AA14ScheduleOID,AA14ScheduleBookingConfig>() {
																						@Override
																						public AA14ScheduleBookingConfig apply(final AA14ScheduleOID schOid) {
																								return AA14ClientAPI.this.schedulesAPI()
																														 .getForCRUD()
																														 .load(schOid)
																														 		.getBookingConfig();
																						}
																			   })
																	.toList();
		// ensure all schedule booking config are compatible
		AA14ScheduleBookingConfig outBookingCfg = CollectionUtils.pickOneElement(bookingCfgs);
		for (AA14ScheduleBookingConfig bookingCfg : bookingCfgs) {
			if (!bookingCfg.hasSameDataAs(outBookingCfg)) throw new IllegalStateException("Error in schedule booking config for location with oid/id=" + loc.getOid() + "/" + loc.getId() +
																						  " NOT all schedulles for this location has the same booking config");
		}
		// return 
		return outBookingCfg;
	}
	private Memoized<Map<AA14OrgDivisionServiceLocationOID,AA14ScheduleBookingConfig>> _scheduleBookingConfigByLocationOid = 
				new Memoized<Map<AA14OrgDivisionServiceLocationOID,AA14ScheduleBookingConfig>>() {
						@Override
						protected Map<AA14OrgDivisionServiceLocationOID,AA14ScheduleBookingConfig> supply() {
							Map<AA14OrgDivisionServiceLocationOID,AA14ScheduleBookingConfig> outMap = Maps.newHashMapWithExpectedSize(_locationsByOid.get().size());
							for (AA14OrgDivisionServiceLocation loc : _locationsByOid.get().values()) {
								outMap.put(loc.getOid(),
										   _locationScheduleBookingConfigFor(loc));
							}
							return outMap;
						}
				};
	private Memoized<Map<AA14OrgDivisionServiceLocationID,AA14ScheduleBookingConfig>> _scheduleBookingConfigByLocationId = 
				new Memoized<Map<AA14OrgDivisionServiceLocationID,AA14ScheduleBookingConfig>>() {
						@Override
						protected Map<AA14OrgDivisionServiceLocationID,AA14ScheduleBookingConfig> supply() {
							Map<AA14OrgDivisionServiceLocationID,AA14ScheduleBookingConfig> outMap = Maps.newHashMapWithExpectedSize(_locationsByOid.get().size());
							for (AA14OrgDivisionServiceLocation loc : _locationsByOid.get().values()) {
								outMap.put(loc.getId(),
										   _locationScheduleBookingConfigFor(loc));
							}
							return outMap;
						}
				};
	/**
	 * Loads the booking config for a certain location
	 *	   BEWARE!	The calendar / booking config is associated with the SCHEDULE, not with the LOCATION or SERVICE,
	 *				any location can be associated with multiple schedules BUT all of them MUST have
	 *				the same booking config
	 * @param locOid locationOid
	 * @return the booking config or null
	 */
	public AA14ScheduleBookingConfig getScheduleBookingConfigFor(final AA14OrgDivisionServiceLocationOID locOid) {
		AA14ScheduleBookingConfig outCfg = _scheduleBookingConfigByLocationOid.get()
																			  .get(locOid);
		if (outCfg == null) log.error("There does NOT exists booking config for location with oid={}",locOid);
		return outCfg;
	}
	/**
	 * Loads the booking config for a certain location
	 *	   BEWARE!	The calendar / booking config is associated with the SCHEDULE, not with the LOCATION or SERVICE,
	 *				any location can be associated with multiple schedules BUT all of them MUST have
	 *				the same booking config
	 * @param locOid locationOid
	 * @return the booking config or null
	 */
	public AA14ScheduleBookingConfig getScheduleBookingConfigFor(final AA14OrgDivisionServiceLocationID locId) {
		AA14ScheduleBookingConfig outCfg = _scheduleBookingConfigByLocationId.get()
																			 .get(locId);
		if (outCfg == null) log.error("There does NOT exists booking config for location with id={}",locId);
		return outCfg;
	}
	/**
	 * Loads the booking config for a certain schedule
	 * @param schOid
	 * @return the booking config or null
	 */
	public AA14ScheduleBookingConfig getScheduleBookingConfigFor(final AA14ScheduleOID schOid) {
		AA14Schedule sch = this.getScheduleFor(schOid);
		return sch.getBookingConfig();
	}
	/**
	 * Loads the booking config for a certain schedule
	 * @param schId
	 * @return the booking config or null
	 */
	public AA14ScheduleBookingConfig getScheduleBookingConfigFor(final AA14ScheduleID schId) {
		AA14Schedule sch = this.getScheduleFor(schId);
		return sch.getBookingConfig();
	}
/////////////////////////////////////////////////////////////////////////////////////////
//	 
/////////////////////////////////////////////////////////////////////////////////////////
	private Memoized<Map<AA14OrgDivisionServiceLocationOID,
					 	 Map<Language,AA14SummarizedOrgHierarchy>>> _summarizedOrgHierarchyByLocationOid = 
				new Memoized<Map<AA14OrgDivisionServiceLocationOID,
					 	 	     Map<Language,AA14SummarizedOrgHierarchy>>>() {
						@Override
						protected Map<AA14OrgDivisionServiceLocationOID, 
									  Map<Language, AA14SummarizedOrgHierarchy>> supply() {
							// Just join all maps
							Map<AA14OrgDivisionServiceLocationOID, 
									  Map<Language, AA14SummarizedOrgHierarchy>> outMap = Maps.newHashMap();
							outMap.putAll(_configForTrafikoa.getSummarizedOrgHierarchyByLocationOid().get());
							outMap.putAll(_configForBizilagun.getSummarizedOrgHierarchyByLocationOid().get());
							outMap.putAll(_configForBloodDonation.getSummarizedOrgHierarchyByLocationOid().get());
							outMap.putAll(_configForMedicalService.getSummarizedOrgHierarchyByLocationOid().get());
							return outMap;
						}
				};
	/**
	 * Loads the location hierarchy by it's oid
	 * @param locOid locationOid
	 * @param lang language code
	 * @return	organization hierarchy
	 */
	public AA14SummarizedOrgHierarchy getOrgHierarchyFor(final AA14OrgDivisionServiceLocationOID locOid,
														 final Language lang) {
		AA14SummarizedOrgHierarchy outHiearchy = _summarizedOrgHierarchyByLocationOid.get()
																					 .get(locOid)
																					 .get(lang);
		if (outHiearchy == null) log.error("There does NOT exists a location with oid={}",locOid);
		return outHiearchy;
	}
	/**
	 * Loads the location hierarchy by it's id
	 * @param locId locationId
	 * @param lang language code
	 * @return	organization hierarchy
	 */
	public AA14SummarizedOrgHierarchy getOrgHierarchyFor(final AA14OrgDivisionServiceLocationID locId,
														 final Language lang) {
		return AA14ConfigBase.filterOrgHierarchyForLocationWithId(_summarizedOrgHierarchyByLocationOid.get(),
																  locId,
																  lang);
	}
}
