package aa14b.services.delegates.notifier;

import java.util.Collection;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.apache.velocity.app.VelocityEngine;

import com.google.common.collect.Maps;

import aa14b.core.config.notifier.AA14ConfigForLatiniaNotifier;
import aa14f.model.AA14NotificationMessage;
import aa14f.model.AA14NotificationOperation;
import aa14f.model.summaries.AA14SummarizedAppointment;
import lombok.extern.slf4j.Slf4j;
import r01f.locale.Language;
import r01f.model.latinia.LatiniaRequestMessage;
import r01f.model.latinia.LatiniaResponse;
import r01f.services.latinia.LatiniaService;
import r01f.types.contact.EMail;
import r01f.types.contact.Phone;
import r01f.util.types.Strings;
import r01f.util.types.collections.CollectionUtils;

/**
 * Latinia notifier, sends SMS to Latinia services.
 */
@Singleton
@Slf4j
public class AA14NotifierServicesLatiniaImpl
     extends AA14NotifierServicesDelegateImplBase {
/////////////////////////////////////////////////////////////////////////////////////////
//  FIELDS
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Latinia services
	 */
	private final LatiniaService _latiniaService;
	
/////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTORS
/////////////////////////////////////////////////////////////////////////////////////////
	@Inject
	public AA14NotifierServicesLatiniaImpl(final AA14ConfigForLatiniaNotifier config,
										   final LatiniaService latiniaService,
										   final VelocityEngine templateEngine) {
		super(config,
			  templateEngine,
			  new AA14NotifierTemplateSelectorLatiniaImpl(config));
		_latiniaService = latiniaService;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  METHODS
/////////////////////////////////////////////////////////////////////////////////////////
	@Override
	public void sendNotification(final AA14NotificationOperation op,
								 final AA14NotificationMessage message) {
		if (_config.isEnabled()) {
			log.info("\t->sending messaging (latinia) notification for appointment with oid={}",
					 message.getAppointment().getOid());
			Collection<Phone> phones = message.getPhonesSanitized();
			if (CollectionUtils.hasData(phones)) {
				// We must send "one message per phone" because when sends Latinia's priority messages, works only with single phone per message.
				for(final Phone currPhone:phones) {
					String body = this.composeMessageBody(op,
														  message.getAppointment(),
														  message.getMails(),message.getPhones());
					if (_config.isSimulated()) {
						log.warn("[SIMULATED!! NOT REALLY SENDING ANY EMAIL (change at aa14b.notifier.properties.xml)][MESSAGING MSG BODY]:");
						log.warn(body);
					}
					else {
						// Transform AA14BAlarmMessage to AA14BLatiniaRequestMessage
						// BEWARE! 	Do NOT set timestamp because the message MUST immediately delivered, 
						//			(timestamp should be used when delivering deferred messages).
						if (!_config.isSimulated()) {
							LatiniaRequestMessage latiniaMsg = new LatiniaRequestMessage();
							latiniaMsg.setAcknowledge("S"); // Telephone company must send acknowledge to latinia service, 
															// AA14B does NOT received this confirmation directly
							latiniaMsg.setMessageContent(body);
							
							// One phone
							latiniaMsg.setReceiverNumbers(currPhone.asStringWithoutCountryCode());
				
							// Send Message to Latinia service
							log.warn("\t--> Sending an instant message to: {}",currPhone.asStringWithoutCountryCode());
							LatiniaResponse response = _latiniaService.sendNotification(latiniaMsg);
							
							//TODO Write response into database
		//					AA14BClientAPI api = AA14BClientAPIProvider.getDefaultApi();
		//					AA14BAlarmEvent alarmEvent = api.alarmEventsAPI()
		//													.getForCRUD()
		//													.load(alarmMessage.getAlarmEventOid());					 
		//					AA14BNotifierResponse alarmResponse = alarmEvent.getAlarmNotificationMsg() != null ? alarmEvent.getAlarmNotificationMsg()
		//																									  : new AA14BNotifierResponse("");
						}
					}
				}
			} else {
				log.warn("\t--> there arent phones to send message...");
			}
		} else {
			log.warn("Instant message sending is DISABLED!");
		}
	}
/////////////////////////////////////////////////////////////////////////////////////////
//	PRIVATE METHODS
/////////////////////////////////////////////////////////////////////////////////////////
	@Override
	protected Map<String,Object> _modelDataFor(final AA14NotificationOperation op,
											   final AA14SummarizedAppointment appointment,
											   final Language lang,
											   final Collection<EMail> mail,final Collection<Phone> phones) {
		// get the model data
	    Map<String,Object> model = super._modelDataFor(op,
	    											   appointment,
	    											   lang,
	    											   mail,phones);
	    // remove accents since SMS service does NOT support them
	    Map<String,Object> outModel = Maps.newHashMapWithExpectedSize(model.size());
	    for (Map.Entry<String,Object> me : model.entrySet()) {
	    	if (me.getValue() == null) continue;
	    	outModel.put(me.getKey(),
	    			     Strings.removeAccents(me.getValue().toString()));
	    }
	    // return
	    return outModel;
	}
}
