package aa14b.services.delegates.notifier;

import java.util.Collection;

import javax.inject.Inject;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import org.apache.velocity.app.VelocityEngine;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.mail.javamail.MimeMessagePreparator;

import aa14b.core.config.notifier.AA14ConfigForMailNotifier;
import aa14f.model.AA14NotificationMessage;
import aa14f.model.AA14NotificationOperation;
import aa14f.model.summaries.AA14SummarizedAppointment;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import r01f.locale.Language;
import r01f.service.ServiceCanBeDisabled;
import r01f.types.contact.EMail;
import r01f.types.contact.Phone;
import r01f.util.types.Strings;
import r01f.util.types.collections.CollectionUtils;

/**
 * EMail notifier
 */
@Slf4j
public class AA14NotifierServicesEMailImpl
     extends AA14NotifierServicesDelegateImplBase {
	
/////////////////////////////////////////////////////////////////////////////////////////
//  INJECTED FIELDS
/////////////////////////////////////////////////////////////////////////////////////////
	@Getter private final JavaMailSender _mailSender;

/////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTOR
/////////////////////////////////////////////////////////////////////////////////////////
	@Inject
	public AA14NotifierServicesEMailImpl(final AA14ConfigForMailNotifier config,
										 final JavaMailSender mailSender,
										 final VelocityEngine templateEngine) {
		super(config,
			  templateEngine,
			  new AA14NotifierTemplateSelectorEMailImpl(config));
		_mailSender = mailSender;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  METHODS
/////////////////////////////////////////////////////////////////////////////////////////
	@Override
	public void sendNotification(final AA14NotificationOperation op,
								 final AA14NotificationMessage message) {
		boolean isEnabled = _config.isEnabled();
		if (_mailSender instanceof ServiceCanBeDisabled) {
			ServiceCanBeDisabled serviceCanBeDisabled = (ServiceCanBeDisabled)_mailSender;
			if (serviceCanBeDisabled.isDisabled()) isEnabled = false;
		}
		if (isEnabled) {
			log.info("\t->sending email notification for appointment with oid={}",
					 message.getAppointment().getOid());
			if (CollectionUtils.hasData(message.getMails())) {
				for (EMail mail : message.getMails()) {
					_sendEMailMessage(op,
									  mail,
									  message.getAppointment(), 
									  message.getMails(),message.getPhones());
				}
			} else {
				log.warn("\t--> there arent eMails to send message...");
			}
		} else {
			log.warn("Mail sending is DISABLED");
		}
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Send a mail message when an alarm message is received
	 * @param op
	 * @param to
	 * @param org
	 * @param log
	 * @param agent
	 */
	private void _sendEMailMessage(final AA14NotificationOperation op,
								   final EMail to,
								   final AA14SummarizedAppointment appointment,
								   final Collection<EMail> emails,final Collection<Phone> phones) {
		log.info("\t-->sending email to {} using {}",
				 to,_mailSender.getClass());
		
		// [1] - Create a MimeMessagePreparator
		final String subject = _composeMailMessageSubject(appointment);
		final String body = this.composeMessageBody(op,
													appointment,
													emails,phones);
		if (_config.isSimulated()) {
			log.warn("[SIMULATED!! NOT REALLY SENDING ANY EMAIL (change at aa14b.notifier.properties.xml)][EMAIL MSG BODY]:");
			log.warn(body);
		} else {
			final EMail from = _config.as(AA14ConfigForMailNotifier.class).getFrom();
			MimeMessagePreparator msgPreparator = new MimeMessagePreparator() {
															@Override
												            public void prepare(final MimeMessage mimeMessage) throws Exception {	
																_createMimeMessageHelper(mimeMessage,
																						 to,
																						 from,
																						 subject,
																						 body);
												            }
												  };
			// [2] - Send the message
	        _mailSender.send(msgPreparator);
		}
	}
	private static MimeMessageHelper _createMimeMessageHelper(final MimeMessage mimeMessage,
													   		  final EMail to,
													   		  final EMail from,
													   		  final String subject,
													   		  final String text) throws MessagingException {
	    MimeMessageHelper message = new MimeMessageHelper(mimeMessage,
	    												  true);	// multi-part!!
	    // To & From
	    message.setTo(to.asString());
	    message.setFrom(from.asString());
	    
	    // Subject
	    message.setSubject(subject);
	    
	    // Text
	    message.setText(text,
	    				true);	// html message
	    
	    // Header image (ALWAYS AFTER message.setText())
//        ClassPathResource res = new ClassPathResource(_msgImage.asRelativeString());
//        message.addInline("logo",res);	
	    return message;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  MAIL MESSAGE COMPOSING
/////////////////////////////////////////////////////////////////////////////////////////	
	private static String _composeMailMessageSubject(final AA14SummarizedAppointment appointment) {
		Language lang = appointment.getPerson().getPreferredLang() != null ? appointment.getPerson().getPreferredLang()
																		   : Language.DEFAULT;
		String outSubject = Strings.customized("{} - {} > {} {}:{}",
											   lang == Language.BASQUE ? "HITZORDUA ESKATZEA" : "SERVICIO DE CITA PREVIA",
											   appointment.getService().getName().toUpperCase(),
											   appointment.getStartDateFormatted(lang),
											   appointment.getStartTime().getHourOfDay(),appointment.getStartTime().getMinuteOfHour());
		
		return outSubject;
	}
}
