package aa14b.calendar;

import java.util.Collection;
import java.util.Date;

import org.joda.time.Days;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;

import aa14f.api.interfaces.AA14FindServicesForBookedSlot;
import aa14f.model.AA14BookedSlot;
import aa14f.model.AA14OrgDivisionServiceLocation;
import aa14f.model.AA14Schedule;
import aa14f.model.AA14ScheduleBookingLimit;
import aa14f.model.timeslots.AA14DayRangeTimeSlots;
import aa14f.model.timeslots.AA14DayTimeSlots;
import aa14f.model.timeslots.AA14TimeSlot;
import aa14f.model.timeslots.AA14TimeSlotsBuilder;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import r01f.model.persistence.FindResult;
import r01f.securitycontext.SecurityContext;
import r01f.types.Range;
import r01f.types.datetime.DayOfMonth;
import r01f.types.datetime.MonthOfYear;
import r01f.types.datetime.Year;
import r01f.util.types.collections.CollectionUtils;
import r01f.util.types.collections.Lists;
import rx.Observable;
import rx.Observer;
import rx.Subscriber;
import rx.functions.Func1;

/**
 * Finds available time slots using db stored data
 */
@Slf4j
@RequiredArgsConstructor
public class AA14CalendarAvailableTimeSlotsDBFindDelegate
     extends AA14CalendarAvailableTimeSlotsFindDelegateBase {
/////////////////////////////////////////////////////////////////////////////////////////
//  
/////////////////////////////////////////////////////////////////////////////////////////
	private final AA14FindServicesForBookedSlot _slotFind;
/////////////////////////////////////////////////////////////////////////////////////////
//  
/////////////////////////////////////////////////////////////////////////////////////////
	@RequiredArgsConstructor
	private class DayBookedSlots {
		private final Year year;
		private final MonthOfYear monthOfYear;
		private final DayOfMonth dayOfMonth;
		private final Collection<AA14BookedSlot> bookedSlots;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//	 
/////////////////////////////////////////////////////////////////////////////////////////
	@Override
	public AA14DayRangeTimeSlots availableTimeSlotsForRange(final SecurityContext securityContext,
												   			final AA14OrgDivisionServiceLocation loc,final AA14Schedule sch,
												   			final Year year,final MonthOfYear monthOfYear,final DayOfMonth dayOfMonth,
												   			final int numberOfDays) {
		// 1) Create a stream of day-available slots
		Observable<AA14DayTimeSlots> rangeAvailableSlots = _createDateRangeAvailableSlotsStream(securityContext,
																								loc,sch,
																								year,monthOfYear,dayOfMonth,
																								numberOfDays);
		// 2) build the AA14DayRangeTimeSlots
		int slotsSizeInMinutes = sch.getBookingConfig().getSlotDefaultLengthMinutes();

		final AA14DayRangeTimeSlots outRangeAvailableSlots = new AA14DayRangeTimeSlots(slotsSizeInMinutes,
																					   year,monthOfYear,dayOfMonth,
																					   numberOfDays);
		rangeAvailableSlots.subscribe(new Observer<AA14DayTimeSlots>() {
											@Override
											public void onNext(final AA14DayTimeSlots thisDaySlots) {
												// add the collection of AA14DayTimeSlots to the out AA14DayRangeTimeSlots
												outRangeAvailableSlots.add(thisDaySlots);
											}
											@Override
											public void onCompleted() {
												// finish
											}
											@Override
											public void onError(final Throwable e) {
												e.printStackTrace(System.out);
											}
									  });
		// 3) return
		// set if there's more & less slots available
		LocalDate initDate = new LocalDate(year.getYear(),monthOfYear.getMonthOfYear(),dayOfMonth.getDayOfMonth());
		LocalDate nextDayDate = new LocalDate(outRangeAvailableSlots.getRequestedDateRange().getUpperBound()).plusDays(1);
		outRangeAvailableSlots.setMoreAvailable(!_isOverTheFutureLimit(nextDayDate,initDate, 
													  				  sch.getBookingConfig().getBookingLimit()));		
		
		return outRangeAvailableSlots;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//	 
/////////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Creates an stream of day-available slots
	 * @param securityContext
	 * @param loc
	 * @param sch
	 * @param year
	 * @param monthOfYear
	 * @param dayOfMonth
	 * @param numberOfDays
	 * @return
	 */
	protected Observable<AA14DayTimeSlots> _createDateRangeAvailableSlotsStream(final SecurityContext securityContext,
																			    final AA14OrgDivisionServiceLocation loc,final AA14Schedule sch,
												   							    final Year year,final MonthOfYear monthOfYear,final DayOfMonth dayOfMonth,
												   							    final int numberOfDays) {
		log.info("Retrieving the bookable slots at locationId={}/scheduleId={} ({} days from {}/{}/{}) from DDBB",
				 loc.getId(),sch.getId(),
				 numberOfDays,
				 year,monthOfYear,dayOfMonth);
		
		// Get a date range BEWARE the limit over which an appointment cannot be booked
		int realNumberOfDaysInRange = _daysInRange(year,monthOfYear,dayOfMonth,
												   numberOfDays,
												   sch.getBookingConfig().getBookingLimit());
		LocalDate startDate = new LocalDate(year.getYear(),monthOfYear.getMonthOfYear(),dayOfMonth.getDayOfMonth());
		final Range<Date> dateRange = Range.closed(startDate.toDate(),
											 	   startDate.plusDays(realNumberOfDaysInRange).toDate());
		
		// use the db stored data to get the available slots
		// 1) get the NON-AVAILABLE slots by schedule
		FindResult<AA14BookedSlot> bookedSlotsFindResult = _slotFind.findRangeBookedSlotsFor(securityContext,
												 										  	 sch.getOid(),	
												 										  	 dateRange);
		final Collection<AA14BookedSlot> bookedSlots = bookedSlotsFindResult.getOrThrow();
		
		// 2) transform the bookedSlots collection (NON-BOOKABLE slots) into a stream of AVAILABLE slots (a AA14DayTimeSlots observable)
		log.debug("...composing the slot calendar for dateRange={}",
				  dateRange);
		Observable<AA14DayTimeSlots> rangeAvailableSlots 
			= Observable.create(
				// 2.1 - create an stream of AA14BookedSlot by day
				new Observable.OnSubscribe<DayBookedSlots>() {
						@Override
						public void call(final Subscriber<? super DayBookedSlots> subscriber) {
							// group the booked slots by date
							LocalDate startDate = new LocalDate(dateRange.getLowerBound());
							LocalDate endDate = new LocalDate(dateRange.getUpperBound());
							LocalDate currDate = startDate;
							for (int i=1; i <= Days.daysBetween(startDate,endDate).getDays(); i++) {
								// within the date-range booked slots, filter the ones at currDate and emit
								DayBookedSlots thisDayBookedSlots = _dayBookedSlotsFiltering(bookedSlots,
																							 currDate);
								subscriber.onNext(thisDayBookedSlots);	// emit
								
								// move to next day
								currDate = currDate.plusDays(1);
							}
							subscriber.onCompleted();
						}
				})
				// 2.2 - transform the stream of DayBookedSlots into an stream of AA14DayTimeSlots: transform from booked slots into available slots
				.map(new Func1<DayBookedSlots,AA14DayTimeSlots>() {
							@Override
							public AA14DayTimeSlots call(final DayBookedSlots dayBookedSlots) {
								log.debug("\t{}-{}-{}",
										  dayBookedSlots.year,dayBookedSlots.monthOfYear,dayBookedSlots.dayOfMonth);
								
								return AA14TimeSlotsBuilder.dayTimeSlotsBuilder(dayBookedSlots.year,
																				dayBookedSlots.monthOfYear,
																				dayBookedSlots.dayOfMonth)
										   .addSlots(// all the day slots (now it's not known if the slot is booked or available)
												   
												   	 // [1] create an observable of bookable time slots 
												   	 // 	(at this moment this observable only emits time-slots: it's not known if they're available or not)
												   	 sch.getBookingConfig().getBookableTimeSlots(sch.getOid())	
												   	 
												   	 // [2] if the slot is contained in one of the day booked slots... it's NOT available
												   	 // 	... unless there can be multiple appointments at the same slot
												   	 	.filter(new Func1<AA14TimeSlot,Boolean>() {
																		@Override
																		public Boolean call(final AA14TimeSlot daySlot) {
																			// if it's saturday or sunday the time slot is NOT available
																			LocalDate thisDay = new LocalDate(dayBookedSlots.year.getYear(),
																											  dayBookedSlots.monthOfYear.getMonthOfYear(),
																											  dayBookedSlots.dayOfMonth.getDayOfMonth());
																			if (thisDay.getDayOfWeek() == 6		// saturday
																			 || thisDay.getDayOfWeek() == 7) {	// sunday
																				return false;
																			}
																			// if it's today, the slots before now are NOT available
																			LocalDate today = new LocalDate();
																			LocalTime now = new LocalTime();
																			if ((thisDay.isEqual(today))
																			 && (now.isAfter(daySlot.getStartTime()))) {
																				return false;
																			}
																			
																			// if ther's no appointments the slot is available
																			if (CollectionUtils.isNullOrEmpty(dayBookedSlots.bookedSlots)) return true;
																			
																			// ...else try to see if there's an appointment overlapping the slot
																			int numAppointmentsInSlot = 0;
																			for (AA14BookedSlot thisDayBookedSlot : dayBookedSlots.bookedSlots) {
																				// booked slot
																				LocalTime  bookedSlotStart = thisDayBookedSlot.getStartTime();
																				LocalTime bookedSlotEnd = thisDayBookedSlot.getEndTime();
																				// slot
																				LocalTime daySlotStart = daySlot.getStartTime();
																				LocalTime daySlotEnd = daySlot.getEndTime();
																				
																				// see if the slot has a booked slot inside
																				//        |================== slot ==================|
																				//		  |--- bookedSlot ---|			   					        [0]
																				//                                |--- bookedSlot ---|              [1] 
																				//               |--- bookedSlot ---|                               [2] 	
																				//                                     |------ bookedSlot ------|   [3]
																				// |------ bookedSlot------|                                        [4]
																				// |----------------------- bookedSlot -------------------------|   [5]
																				
																				// TODO multiple events at a slot																			
																				if ((bookedSlotStart.isEqual(daySlotStart))											// [0] 
																				 || (bookedSlotEnd.isEqual(daySlotEnd))												// [1]
																				 || (bookedSlotStart.isAfter(daySlotStart) && bookedSlotEnd.isBefore(daySlotEnd))   // [2]
																				 || (bookedSlotStart.isAfter(daySlotStart) && bookedSlotStart.isBefore(daySlotEnd)) // [3]
																				 || (bookedSlotEnd.isAfter(daySlotStart) && bookedSlotEnd.isBefore(daySlotEnd))     // [4]
																				 || (bookedSlotStart.isBefore(daySlotStart) && bookedSlotEnd.isAfter(daySlotEnd))) {// [5]
																					numAppointmentsInSlot++;
																				} 
																			}
																			// check if there's less than the max number of appointments
																			int maxAppointmentsInSlot = sch.getBookingConfig().getMaxAppointmentsInSlot();
																			boolean isAvailable = maxAppointmentsInSlot > numAppointmentsInSlot;
																			
																			log.debug("\t\t...{} - {} available: {}",
																					  daySlot.getStartTime(),daySlot.getEndTime(),
																					  isAvailable);
																			return isAvailable;
																		}
													 		 }))
										   .build();
								}
					}	// Func1
		);
		
		return rangeAvailableSlots;
	}
	/**
	 * Given a db-stored booked slots within a date range this method filters the booked slots at an also given day
	 * @param rangeBookedSlots
	 * @param date
	 * @return
	 */
	private DayBookedSlots _dayBookedSlotsFiltering(final Collection<AA14BookedSlot> rangeBookedSlots,
												    final LocalDate date) {
		DayBookedSlots dayBookedSlots = new DayBookedSlots(Year.of(date),MonthOfYear.of(date),DayOfMonth.of(date),
														   Lists.<AA14BookedSlot>newArrayList());
		if (CollectionUtils.hasData(rangeBookedSlots)) {
			// filter this day db-stored booked slots (within all booked slots in the given range, filter the ones at this day)
			for (AA14BookedSlot bookedSlot : rangeBookedSlots) {
				LocalDate bookedSlotDate = new LocalDate(bookedSlot.getStartDate());
				if (bookedSlotDate.isEqual(date)) {
					dayBookedSlots.bookedSlots.add(bookedSlot);
				}
			}
		}
		return dayBookedSlots;
	}
	/**
	 * Returns the days in the date range having into account the booking date limits
	 * @return
	 */
	private static int _daysInRange(final Year year,final MonthOfYear monthOfYear,final DayOfMonth dayOfMonth,
							 		final int numberOfDays,
							 		final AA14ScheduleBookingLimit bookingLimit) {
		// BEWARE the limit over which an appointment cannot be booked
		int realNumberOfDaysInRange = numberOfDays;
		LocalDate startDate = new LocalDate(year.getYear(),monthOfYear.getMonthOfYear(),dayOfMonth.getDayOfMonth());
		if (bookingLimit != null) {
			// get the date limit
			LocalDate startDatePlusLimit = bookingLimit != null && bookingLimit.getDaysInFutureLimit() > 0
												? startDate.plusDays(bookingLimit.getDaysInFutureLimit())
												: null;
			LocalDate configuredDateLimit = bookingLimit != null && bookingLimit.getDateLimit() != null
												? new LocalDate(bookingLimit.getDateLimit())
												: null;
			LocalDate dateLimit = null;
			if (startDatePlusLimit != null && configuredDateLimit != null) {
				dateLimit = startDatePlusLimit.isBefore(configuredDateLimit) ? startDatePlusLimit : configuredDateLimit;
			} else if (startDatePlusLimit != null) {
				dateLimit = startDatePlusLimit;
			} else if (configuredDateLimit != null) {
				dateLimit = configuredDateLimit;
			}
			
			if (dateLimit != null 
			 && dateLimit.isAfter(startDate)) {
				int daysToLimit = Days.daysBetween(startDate,dateLimit)
									  .getDays();
				if (daysToLimit < numberOfDays) {
					log.info("...requested avaliable slots for {} days from {} BUT the limit is {} days ahead at {}: only slots for {} days will be returned!",
						     numberOfDays,startDate,
						     daysToLimit,dateLimit,
						     daysToLimit);
					realNumberOfDaysInRange = daysToLimit;
				}
			}
		}
		return realNumberOfDaysInRange;
	}
}
