package aa14b.bootstrap.core;

import java.io.IOException;
import java.util.Properties;

import javax.inject.Provider;

import org.apache.velocity.app.VelocityEngine;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.ui.velocity.VelocityEngineFactory;
import org.springframework.ui.velocity.VelocityEngineFactoryBean;

import com.google.inject.Binder;
import com.google.inject.Module;
import com.google.inject.Provides;
import com.google.inject.Singleton;

import aa14b.core.config.notifier.AA14ConfigForLatiniaNotifier;
import aa14b.core.config.notifier.AA14ConfigForLogNotifier;
import aa14b.core.config.notifier.AA14ConfigForMailNotifier;
import aa14b.core.config.notifier.AA14ConfigForNotifierScheduler;
import aa14b.core.config.notifier.AA14ConfigForTwilioNotifier;
import aa14b.notifier.scheduler.quartz.AA14NotifierMockSchedulerWhenDisabled;
import aa14b.notifier.scheduler.quartz.AA14NotifierQuartzSchedulerWrapper;
import aa14b.notifier.scheduler.quartz.AA14NotifierScheduler;
import aa14b.services.delegates.notifier.AA14NotifierServicesDelegateImpl;
import aa14b.services.delegates.notifier.AA14NotifierServicesEMailImpl;
import aa14b.services.delegates.notifier.AA14NotifierServicesLatiniaImpl;
import aa14b.services.delegates.notifier.AA14NotifierServicesLoggerImpl;
import aa14b.services.delegates.notifier.AA14NotifierServicesVoiceImpl;
import aa14f.common.internal.AA14AppCodes;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;
import r01b.bootstrap.scheduler.QuartzSchedulerGuiceModule;
import r01f.guids.CommonOIDs.AppComponent;
import r01f.mail.JavaMailSenderProvider;
import r01f.notifier.UseEMailNotifier;
import r01f.notifier.UseLogNotifier;
import r01f.notifier.UseMessagingNotifier;
import r01f.notifier.UseVoiceNotifier;
import r01f.services.latinia.LatiniaService;
import r01f.services.latinia.LatiniaServiceAPIData;
import r01f.twilio.TwilioService;
import r01f.twilio.TwilioServiceProvider;
import r01f.xmlproperties.XMLProperties;
import r01f.xmlproperties.XMLPropertiesComponent;
import r01f.xmlproperties.XMLPropertiesForAppComponent;

@Slf4j
@EqualsAndHashCode				// This is important for guice modules
     class AA14NotifierGuiceModule 
implements Module {
/////////////////////////////////////////////////////////////////////////////////////////
//  FIELDS
/////////////////////////////////////////////////////////////////////////////////////////
	private final AA14ConfigForNotifierScheduler _notifierSchedulerConfig;
	
	private final AA14ConfigForMailNotifier _mailNotifierConfig;
	private final AA14ConfigForLatiniaNotifier _latiniaNotifierConfig;
	private final AA14ConfigForTwilioNotifier _twilioNotifierConfig;
	private final AA14ConfigForLogNotifier _loggerNotifierConfig;
/////////////////////////////////////////////////////////////////////////////////////////
//  CONSTRUCTOR
/////////////////////////////////////////////////////////////////////////////////////////
	public AA14NotifierGuiceModule(// notifier scheduler
								   final AA14ConfigForNotifierScheduler notifierSchedulerConfig,
								   // notifiers
								   final AA14ConfigForMailNotifier mailNotifierConfig,
								   final AA14ConfigForLatiniaNotifier latiniaNotifierConfig,
								   final AA14ConfigForTwilioNotifier twilioNotifierConfig,
								   final AA14ConfigForLogNotifier loggerNotifierConfig) {
		_notifierSchedulerConfig = notifierSchedulerConfig;
		
		_mailNotifierConfig = mailNotifierConfig;
		_latiniaNotifierConfig = latiniaNotifierConfig;
		_twilioNotifierConfig = twilioNotifierConfig;
		_loggerNotifierConfig = loggerNotifierConfig;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  MODULE
/////////////////////////////////////////////////////////////////////////////////////////	
	@Override
	public void configure(final Binder binder) {
		////////// Notifier scheduler
		_bindScheduler(binder);
		
		////////// Notifiers
		// [0] - Log (event notifier services that just logs the event)
		binder.bind(AA14NotifierServicesDelegateImpl.class)
			  .annotatedWith(UseLogNotifier.class)
			  .to(AA14NotifierServicesLoggerImpl.class)
			  .in(Singleton.class);

		// [1] - EMail (see provider method below)
		binder.bind(AA14NotifierServicesDelegateImpl.class)
			  .annotatedWith(UseEMailNotifier.class)
			  .to(AA14NotifierServicesEMailImpl.class)		// gets injected with a java mail sender (see provider below)
			  .in(Singleton.class);	
		
		// [2] - Latinia (see provider method below)
		binder.bind(AA14NotifierServicesDelegateImpl.class)
			  .annotatedWith(UseMessagingNotifier.class)
			  .to(AA14NotifierServicesLatiniaImpl.class)	// gets injected with Latinia service (see provider below)
			  .in(Singleton.class);
		
		// [3] - Voice (see provider method below)
		binder.bind(AA14NotifierServicesDelegateImpl.class)
			  .annotatedWith(UseVoiceNotifier.class)
			  .to(AA14NotifierServicesVoiceImpl.class)		// gets injected with Twilio service (see provider below)
			  .in(Singleton.class);
				
		////////// Velocity engine to create the messages
		binder.bind(VelocityEngine.class)
			  .toProvider(new Provider<VelocityEngine>() {
									@Override
									public VelocityEngine get() {
										Properties velocityProps = new Properties();
										velocityProps.put("resource.loader","class");
								        velocityProps.put("class.resource.loader.class","org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader");
										
										VelocityEngineFactory velocityEngineFactory = new VelocityEngineFactoryBean();
										velocityEngineFactory.setVelocityProperties(velocityProps);
										VelocityEngine outVelocityEngine = null;
										try  {
											outVelocityEngine = velocityEngineFactory.createVelocityEngine();
										} catch(IOException ioEx) {
											ioEx.printStackTrace(System.out);
										}
										return outVelocityEngine;
									}
						  })
			  .in(Singleton.class);
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  PROPERTIES FOR PROVIDERS
/////////////////////////////////////////////////////////////////////////////////////////
	@Provides @XMLPropertiesComponent("notifier")
	XMLPropertiesForAppComponent provideXMLPropertiesForServices(final XMLProperties props) {
		XMLPropertiesForAppComponent outPropsForComponent = new XMLPropertiesForAppComponent(props.forApp(AA14AppCodes.CORE_APPCODE),
																							 AppComponent.forId("notifier"));
		return outPropsForComponent;
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  NOTIFIER SERVICE PROVIDER
/////////////////////////////////////////////////////////////////////////////////////////
	@Provides @Singleton 	// provides a single instance of the java mail sender
	AA14NotifierServicesEMailImpl _provideMailNotifier(final VelocityEngine velocityEngine) {
		JavaMailSenderProvider javaMailSenderProvider = new JavaMailSenderProvider(_mailNotifierConfig.getMailSenderConfig());
		JavaMailSender javaMailSender = javaMailSenderProvider.get();
		return new AA14NotifierServicesEMailImpl(_mailNotifierConfig,
												 javaMailSender,
												 velocityEngine);
	}
	/**
	 * Provides a {@link LatiniaService} implementation
	 * @param props
	 * @return
	 */
	@Provides @Singleton	// provides a single instance of the latinia service
	AA14NotifierServicesLatiniaImpl _provideLatiniaNotifier(final VelocityEngine velocityEngine) {
		// Provide a new latinia service api data using the provider
		LatiniaServiceAPIData latiniaApiData = _latiniaNotifierConfig.getLatiniaConfig();
		// Using the latinia service api data create the LatiniaService object
		LatiniaService latiniaService = new LatiniaService(latiniaApiData);
		return new AA14NotifierServicesLatiniaImpl(_latiniaNotifierConfig,
												   latiniaService,
												   velocityEngine);
	}	
	/**
	 * Provides a {@link TwilioService} implementation
	 * @param props
	 * @return
	 */
	@Provides @Singleton	// provides a single instance of the twilio service
	AA14NotifierServicesVoiceImpl _provideTwilioNotifier(final VelocityEngine velocityEngine) {
		TwilioServiceProvider twilioServiceProvider = new TwilioServiceProvider(_twilioNotifierConfig.getTwilioConfig());
		TwilioService twilioService = twilioServiceProvider.get();
		return new AA14NotifierServicesVoiceImpl(_twilioNotifierConfig,
										   		 twilioService,
										         velocityEngine);
	}
	/**
	 * Provides a {@link TwilioService} implementation
	 * @param props
	 * @return
	 */
	@Provides @Singleton	// provides a single instance of the twilio service
	AA14NotifierServicesLoggerImpl _provideLoggerNotifier(final VelocityEngine velocityEngine) {
		return new AA14NotifierServicesLoggerImpl(_loggerNotifierConfig,
												 velocityEngine);
	}
/////////////////////////////////////////////////////////////////////////////////////////
//  BIND SCHEDULER
/////////////////////////////////////////////////////////////////////////////////////////
	private void _bindScheduler(final Binder binder) {		
		// scheduler config
		log.warn("[Notifier Scheduler Config:]\n{}",
				 _notifierSchedulerConfig.debugInfo());
		binder.bind(AA14ConfigForNotifierScheduler.class)
			  .toInstance(_notifierSchedulerConfig);
		
		// scheduler
		if (_notifierSchedulerConfig.getSchedulerConfig().isEnabled()) {
			// quartz 
			binder.install(new QuartzSchedulerGuiceModule(_notifierSchedulerConfig.getSchedulerConfig()));
			
			// render scheduler
			binder.bind(AA14NotifierScheduler.class)
				  .to(AA14NotifierQuartzSchedulerWrapper.class)
				  .in(Singleton.class);
			
		} else {
			// bind a mock impl 
			binder.bind(AA14NotifierScheduler.class)
				  .to(AA14NotifierMockSchedulerWhenDisabled.class)
				  .in(Singleton.class);
			
		}
	}
}
