package com.gfi.utils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

/**
 * Utilidades para el manejo de XMLs
 * 
 * @author GFI-NORTE
 * 
 */
public class UtilXML {

	private static final Logger logger = LoggerFactory.getLogger(UtilXML.class);

	/**
	 * Constructor privado
	 */
	private UtilXML() {
	}

	/**
	 * Obtiene el texto de un XML(Node o Document) en formato String
	 * 
	 * @param node
	 *            Node
	 * @return String
	 */
	public static String xml2String(Node node) {
		UtilXML.logger.trace("xml2String INI");
		try {
			Source source = new DOMSource(node);
			StringWriter stringWriter = new StringWriter();
			Result result = new StreamResult(stringWriter);
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();
			transformer.transform(source, result);
			String resultado = stringWriter.getBuffer().toString();
			UtilXML.logger.trace("xml2String FIN");
			return resultado;
		} catch (TransformerConfigurationException e) {
			UtilXML.logger.error("xml2String", e);
		} catch (TransformerException e) {
			UtilXML.logger.error("xml2String", e);
		}
		return null;
	}

	/**
	 * Obtiene un objeto Node(XML) a partir de un InputStream
	 * 
	 * @param input
	 *            InputStream
	 * @return Document el document
	 */
	public static Node inputStream2Xml(InputStream input) {
		UtilXML.logger.trace("inputStream2Xml INI");
		Document tmpX = null;
		DocumentBuilder builder = null;
		try {
			builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			tmpX = builder.parse(input);
			tmpX.getDocumentElement().normalize();
		} catch (ParserConfigurationException e) {
			UtilXML.logger.error("inputStream2Xml", e);
		} catch (SAXException e) {
			UtilXML.logger.error("inputStream2Xml", e);
		} catch (IOException e) {
			UtilXML.logger.error("inputStream2Xml", e);
		}

		UtilXML.logger.trace("inputStream2Xml FIN");
		return tmpX;

	}

	/**
	 * Obtiene un objeto Node(XMl) a partir de una cadena.
	 * 
	 * @param str
	 *            la cadena
	 * @return Node el documento
	 */
	public static Node string2Xml(String str) {
		UtilXML.logger.trace("string2Xml INI");
		InputStream is = new ByteArrayInputStream(str.getBytes());
		Node result = null;
		try {
			result = UtilXML.inputStream2Xml(is);
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				UtilXML.logger.warn("Error al cerrar inputStream.",
						e.getMessage());
			}
		}
		UtilXML.logger.trace("string2Xml FIN");
		return result;

	}

	/**
	 * Valida un XML contra un esquema
	 * 
	 * @param inXSD
	 *            InputStream
	 * @param dXML
	 *            Node
	 * @return boolean
	 */
	public static boolean validateSchema(InputStream inXSD, Node dXML) {
		UtilXML.logger.trace("validateXMLSchema INI");
		boolean resultado = false;
		try {
			SchemaFactory factory = SchemaFactory
					.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
			Schema schema = factory.newSchema(new StreamSource(inXSD));
			Validator validator = schema.newValidator();
			validator.validate(new DOMSource(dXML));
			resultado = true;
		} catch (IOException e) {
			UtilXML.logger.warn("validateXMLSchema", e);
		} catch (SAXException e) {
			UtilXML.logger.warn("validateXMLSchema", e);
		}
		UtilXML.logger.trace("validateXMLSchema resultado: {}", resultado);
		return resultado;
	}

}
