package com.gfi.beans;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;

import javax.activation.MimetypesFileTypeMap;

/**
 * Clase que contiene los datos de un fichero guardado en disco.<br/>
 * Mantiene por un lado el nombre original que se le ha querido dar la fichero,
 * y la ruta del fichero temporal que almacena su información.
 * 
 * @author GFI
 */
public class FileObject implements Serializable {

	private static final long serialVersionUID = 1L;

	private String originalName;
	private File fileTemp;

	/**
	 * Crea un nuevo fichero, especificando su nombre completo, y la carpeta de
	 * servidor donde se contiene.
	 * 
	 * @param name
	 *            String con el nombre y la extension del fichero
	 * @param pathFolder
	 *            String
	 * @throws IOException
	 *             e
	 */
	public FileObject(String name, String pathFolder) throws IOException {
		int indexExt = name.lastIndexOf(".");
		String ext = name.substring(indexExt);
		String nameWithoutExt = name.substring(0, indexExt);
		this.originalName = name;
		this.fileTemp = File.createTempFile(nameWithoutExt, ext, new File(
				pathFolder));
	}

	/**
	 * Crea un nuevo fichero, especificando su nombre, extension, y la carpeta
	 * de servidor donde se contiene.
	 * 
	 * @param nameWithoutExt
	 *            String con el nombre del fichero
	 * @param ext
	 *            String con la extension del fichero
	 * @param pathFolder
	 *            String
	 * @throws IOException
	 *             e
	 */
	public FileObject(String nameWithoutExt, String ext, String pathFolder)
			throws IOException {
		StringBuilder sb = new StringBuilder();
		sb.append(nameWithoutExt).append(".").append(ext);
		this.originalName = sb.toString();
		this.fileTemp = File.createTempFile(nameWithoutExt, "." + ext,
				new File(pathFolder));
	}

	/**
	 * @return String
	 */
	public String getOriginalName() {
		return this.originalName;
	}

	/**
	 * Devuelve el InputStream de lectura del fichero
	 * 
	 * @return InputStream
	 * @throws FileNotFoundException
	 *             e
	 */
	public InputStream getInputStream() throws FileNotFoundException {
		return new FileInputStream(this.fileTemp);
	}

	/**
	 * Devuelve el InputStream de lectura del fichero
	 * 
	 * @return InputStream
	 * @throws FileNotFoundException
	 *             e
	 */
	public OutputStream getOutputStream() throws FileNotFoundException {
		return new FileOutputStream(this.fileTemp);
	}

	/**
	 * 
	 * @return Tamaño del fichero en disco.
	 */
	public Long getSize() {
		return this.fileTemp.length();
	}

	/**
	 * 
	 * @return Mimetype del archivo
	 */
	public String getMimeType() {
		return new MimetypesFileTypeMap().getContentType(this.fileTemp);
	}
}