package eu.europa.ec.publications.esentool.rest.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import javax.xml.bind.DatatypeConverter;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Consts;
import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.util.ISO8601DateFormat;

public class EsentoolRestClient {
    private HttpClientBuilder clientBuilder;
    private ObjectMapper mapper = new ObjectMapper();
    private final String url;
    private final String username;
    private final String password;

    private EsentoolRestClient(String url, String username, String password) {
        this.url = url;
        this.username = username;
        this.password = password;
        mapper.setDateFormat(new ISO8601DateFormat());
        mapper.setPropertyNamingStrategy(new PropertyNamingStrategy.LowerCaseWithUnderscoresStrategy());
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * This method allows to submit a new notice.
     *
     * @param notice the notice as XML encoded in base64
     * @return a {@link NoticeInformation} object containing information related to the notice submitted
     */
    public NoticeInformation submitNotice(byte[] notice) {
        HttpPost post = post("/latest/notice/submit");
        List<NameValuePair> formparams = new ArrayList<NameValuePair>();
        formparams.add(new BasicNameValuePair("notice", new String(notice)));
        post.setEntity(new UrlEncodedFormEntity(formparams, Consts.UTF_8));
        return execute(post, new TypeReference<NoticeInformation>() {
        });
    }

    /**
     * This method allows to retrieve information related to a given notice.
     *
     * @param submissionId the submission ID of the notice
     * @return a {@link NoticeInformation} object containing information related to the notice given as parameter
     */
    public NoticeInformation getNoticeInformation(String submissionId) {
        HttpGet get = get("/latest/notice/" + submissionId, null);
        return execute(get, new TypeReference<NoticeInformation>() {
        });
    }

    /**
     * This method allows to render a notice as XML encoded in base64 in PDF as displayed
     * on TED website, in HTML as displayed on TED website, or in PDF as displayed in
     * eNotices.
     *
     * @param notice the notice as XML encoded in base64
     * @param format one of the value of {@link RenderFormat}
     * @param language the language of the rendering. The language must be specified in ISO2. For examples: FR, DE, EN ...
     * @return the notice rendered into the {@code RenderFormat} given as parameter
     */
    public byte[] renderNotice(byte[] notice, RenderFormat format, String language) {
        HttpPost post = post("/latest/notice/render");
        List<NameValuePair> formparams = new ArrayList<NameValuePair>();
        formparams.add(new BasicNameValuePair("notice", new String(notice)));
        formparams.add(new BasicNameValuePair("format", format.name()));
        formparams.add(new BasicNameValuePair("language", language));
        post.setEntity(new UrlEncodedFormEntity(formparams, Consts.UTF_8));
        SimpleResult result = execute(post, new TypeReference<SimpleResult>() {
        });
        try {
            return result.getResult().getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * This method allows to render a notice as XML encoded in base64 in PDF as displayed
     * on TED website, in HTML as displayed on TED website, or in PDF as displayed in
     * eNotices.
     *
     * @param submissionId the submission ID of the notice to be rendered
     * @param format one of the value of {@link RenderFormat}
     * @param language the language of the rendering. The language must be specified in ISO2. For examples: FR, DE, EN ...
     * @return the notice rendered into the {@code RenderFormat} given as parameter
     */
    public byte[] renderNotice(String submissionId, RenderFormat format, String language) {
        HttpPost post = post("/latest/notice/render");
        List<NameValuePair> formparams = new ArrayList<NameValuePair>();
        formparams.add(new BasicNameValuePair("submission_id", submissionId));
        formparams.add(new BasicNameValuePair("format", format.name()));
        formparams.add(new BasicNameValuePair("language", language));
        post.setEntity(new UrlEncodedFormEntity(formparams, Consts.UTF_8));
        SimpleResult result = execute(post, new TypeReference<SimpleResult>() {
        });
        try {
            return result.getResult().getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * This method allows to search for notices according to specific parameters.
     *
     * @param params a {@link SearchNoticeParams} object to refine search
     * @return a {@code Page} of {@link NoticeInformation} objects related to the search criteria given as parameter
     */
    public Page<NoticeInformation> searchNotices(SearchNoticeParams params) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd");
        sdf.setLenient(false);
        List<NameValuePair> nvps = new ArrayList<NameValuePair>();
        nvps.add(new BasicNameValuePair("page", String.valueOf(params.getPage())));
        nvps.add(new BasicNameValuePair("pageSize", String.valueOf(params.getPageSize())));
        if (params.getNoticeStatus() != null) {
            nvps.add(new BasicNameValuePair("status", params.getNoticeStatus().name()));
        }
        if (params.getReceivedFrom() != null) {
            nvps.add(new BasicNameValuePair("receivedFrom", sdf.format(params.getReceivedFrom())));
        }
        if (params.getReceivedTo() != null) {
            nvps.add(new BasicNameValuePair("receivedTo", sdf.format(params.getReceivedTo())));
        }
        if (params.getSort() != null) {
            nvps.add(new BasicNameValuePair("sort", params.getSort() + "," + params.getSortDirection().name()));
        }
        HttpGet get = get("/latest/notice/search", nvps);
        return execute(get, new TypeReference<Page<NoticeInformation>>() {
        });
    }

    private <T> T execute(HttpRequestBase request, TypeReference type) {
        CloseableHttpClient httpClient = null;
        try {
            try {
                httpClient = clientBuilder.build();
                HttpResponse response = httpClient.execute(request);
                return parse(response, type);
            } catch (IOException e) {
                throw new RestClientException(e);
            }
        } finally {
            if (httpClient != null) {
                try {
                    httpClient.close();
                } catch (IOException e) {
                    //
                }
            }
        }
    }

    private HttpPost post(String path) {
        URI uri = getUri(path, null);
        HttpPost request = new HttpPost(uri);
        buildHeaders(request);
        return request;
    }

    private HttpGet get(String path, List<NameValuePair> params) {
        URI uri = getUri(path, params);
        HttpGet request = new HttpGet(uri);
        buildHeaders(request);
        return request;
    }

    private URI getUri(String path, List<NameValuePair> params) {
        URI uri;
        try {
            final URIBuilder uriBuilder = new URIBuilder(new URI(url + path));
            if (params != null) {
                uriBuilder.setParameters(params);
            }
            uri = uriBuilder.build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
        return uri;
    }

    private void buildHeaders(HttpRequestBase request) {
        try {
            request.addHeader("Authorization", "Basic " + DatatypeConverter.printBase64Binary((username + ":" + password).getBytes("UTF-8")));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        request.addHeader("Accept", "application/json");
    }

    private <T> T parse(HttpResponse response, TypeReference typeReference) throws IOException {
        InputStream is = response.getEntity().getContent();
        handleError(response, is);
        return mapper.readValue(is, typeReference);
    }

    private void handleError(HttpResponse response, InputStream is) throws IOException {
        if (response.getStatusLine().getStatusCode() >= 400) {
            Header contentTypeHeader = response.getFirstHeader("Content-Type");
            if (contentTypeHeader != null && StringUtils.startsWithIgnoreCase(contentTypeHeader.getValue(), "application/json")) {
                Error error = mapper.readValue(is, Error.class);
                throw new RestClientException(error);
            } else {
                throw new RuntimeException(response.getStatusLine().toString());
            }
        }
    }

    public static class EsentoolRestClientBuilder {
        private final String url;
        private final String username;
        private final String password;
        private ProxySettings proxySettings;
        private HttpClientBuilder clientBuilder;
        private ObjectMapper objectMapper;

        /**
         * A builder to create an {@link eu.europa.ec.publications.esentool.rest.client.EsentoolRestClient} instance.
         *
         * @param url the url where the client is host
         * @param username the <code>eSender login</code> or the <code>customer login</code> preceded by its related eSender login
         * @param password the related eSender's web service password
         */
        public EsentoolRestClientBuilder(String url, String username, String password) {
            this.url = url;
            this.username = username;
            this.password = password;
        }

        public ProxySettings throughProxy() {
            final ProxySettings proxySettings = new ProxySettings();
            this.proxySettings = proxySettings;
            return proxySettings;
        }

        public EsentoolRestClientBuilder withHttpClient(HttpClientBuilder clientBuilder) {
            this.clientBuilder = clientBuilder;
            return this;
        }

        /**
         * Provide a custom {@link ObjectMapper} to use to deserialize the response.
         * It is recommended to set at least the three following properties
         * <pre>
         * mapper.setDateFormat(new ISO8601DateFormat());
         * mapper.setPropertyNamingStrategy(new PropertyNamingStrategy.LowerCaseWithUnderscoresStrategy());
         * mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
         * </pre>
         * @param objectMapper
         * @return
         */
        public EsentoolRestClientBuilder withObjectMapper(ObjectMapper objectMapper) {
            this.objectMapper = objectMapper;
            return this;
        }

        public EsentoolRestClient build() {
            EsentoolRestClient esentoolRestClient = new EsentoolRestClient(url, username, password);
            esentoolRestClient.clientBuilder = (clientBuilder == null) ? HttpClients.custom() : clientBuilder;
            if (objectMapper != null) {
                esentoolRestClient.mapper = objectMapper;
            }
            if (proxySettings != null) {
                if (proxySettings.proxyUsername != null) {
                    final BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
                    credentialsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(proxySettings.proxyUsername, proxySettings.proxyPassword));
                    esentoolRestClient.clientBuilder.setDefaultCredentialsProvider(credentialsProvider);
                }
                if (proxySettings.hostname != null) {
                    esentoolRestClient.clientBuilder.setProxy(new HttpHost(proxySettings.hostname, proxySettings.port));
                }
            }
            return esentoolRestClient;
        }

        public class ProxySettings {
            private String hostname;
            private int port;
            private String proxyUsername;
            private String proxyPassword;

            public ProxySettings withHostname(String hostname) {
                this.hostname = hostname;
                return this;
            }

            public ProxySettings withPort(int port) {
                this.port = port;
                return this;
            }

            public ProxySettings withUsername(String proxyUsername) {
                this.proxyUsername = proxyUsername;
                return this;
            }

            public ProxySettings withPassword(String proxyPassword) {
                this.proxyPassword = proxyPassword;
                return this;
            }
        }
    }
}
